package com.bxm.adx.common.sell;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.buyers.BuyerWrapper;
import com.bxm.adx.common.buy.buyers.PriorityBuyers;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.optimization.Optimization;
import com.bxm.adx.common.buy.optimization.OptimizationDao;
import com.bxm.adx.common.ip.IpService;
import com.bxm.adx.common.market.MarketOrders;
import com.bxm.adx.common.market.MarketRequest;
import com.bxm.adx.common.openlog.event.internal.MediaRequestEvent;
import com.bxm.adx.common.sell.init.BidRequestInitializer;
import com.bxm.adx.common.sell.init.InitializerParam;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.position.PositionService;
import com.bxm.adx.common.sell.request.App;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.common.sell.request.Impression;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.user.id.generator.DeviceHelper;
import com.bxm.user.id.generator.DeviceInfo;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.ip.IP;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;

/**
 * @author allen
 * @since 2019-12-20
 */
@Slf4j
@Configuration
public class DefaultBidRequestWrapper implements BidRequestWrapper, ApplicationListener<ApplicationReadyEvent> {

    private final PriorityBuyers priorityBuyers;
    private final PositionService positionService;
    private final AdxProperties properties;;
    private final EventPark eventPark;
    private List<BidRequestInitializer> initializers = Lists.newArrayList();

    public DefaultBidRequestWrapper(PriorityBuyers priorityBuyers, PositionService positionService,
                                    AdxProperties properties, EventPark eventPark) {
        this.priorityBuyers = priorityBuyers;
        this.positionService = positionService;
        this.properties = properties;
        this.eventPark = eventPark;
    }

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        initializers.addAll(event.getApplicationContext().getBeansOfType(BidRequestInitializer.class).values());
        initializers.sort(Comparator.comparingInt(Ordered::getOrder));
    }

    @Override
    public MarketOrders packing(BidRequest bidRequest) {
        if (null == bidRequest) {
            return null;
        }
        List<Impression> imps = bidRequest.getImps();
        if (CollectionUtils.isEmpty(imps)) {
            throw new AdxException(AdxErrEnum.POSITION_NOT_FOUND);
        }
        AdxContextFactory.get().setBidRequest(bidRequest);

        Impression impression = bidRequest.getImps().iterator().next();
        //映射广告位
        Position mediaPosition = getPositionByBidRequest(bidRequest, impression);

        Map<Position, List<List<BuyerWrapper>>> positionListMap = Maps.newHashMap();
        List<Position> positionList = new ArrayList<>();

        if (Objects.nonNull(mediaPosition) && mediaPosition.isEnabled()) {
            positionList.add(mediaPosition);
            AdxContextFactory.get().setPosition(mediaPosition);
            //检查填充请求需要的默认参数
            initializers.forEach(init -> init.accept(bidRequest, InitializerParam.builder().position(mediaPosition).build()));
            // 查找这些广告位分配的卖家和等级
            List<List<BuyerWrapper>> buyers =
                    priorityBuyers.findAsPriority(mediaPosition, impression.getBid_floor(), impression.getBid_top());
            priorityBuyers.rebuildBuyers(bidRequest, mediaPosition, buyers);
            positionListMap.put(mediaPosition, buyers);
        }

        if (CollectionUtils.isEmpty(positionListMap)) {
            throw new AdxException(AdxErrEnum.POSITION_NOT_FOUND);
        }

        BidConfig bidConfig = new BidConfig()
                .setDisDot(Math.abs(bidRequest.getId().hashCode() % 100) >= properties.getDotSimplingPercent());
        AdxContextFactory.get().setBidConfig(bidConfig);
        //埋点媒体请求
        eventPark.post(new MediaRequestEvent(this, bidRequest, bidConfig));

        // 索引即表示优先级，从0开始。
        List<MarketRequest> requests = Lists.newArrayListWithCapacity(properties.getMaxCapacityOfBuyers());
        // 重组，分出每一个优先级中对应的DSP+广告位
        for (Map.Entry<Position, List<List<BuyerWrapper>>> positionListEntry : positionListMap.entrySet()) {
            Position position = positionListEntry.getKey();
            List<List<BuyerWrapper>> list = positionListEntry.getValue();
            int level = list.size();
            if (level == 0) {
                // 没有设置分流策略
                continue;
            }

            for (int i = 0; i < level; i++) {
                // 这个广告位第i级的DSP配置列表
                List<BuyerWrapper> array = list.get(i);
                if (CollectionUtils.isEmpty(array)) {
                    continue;
                }
                MarketRequest marketRequest = CollectionUtils.isEmpty(requests) ? null : requests.size() > i ? requests.get(i) : null;
                if (null == marketRequest) {
                    marketRequest = new MarketRequest(i);
                    requests.add(i, marketRequest);
                }
                marketRequest.addBuyers(array);
                marketRequest.addPosition(position);
            }
        }

        MarketOrders marketOrders = new MarketOrders(bidRequest, requests, positionList, getUid(bidRequest));
        if (log.isDebugEnabled()) {
            log.debug("{}", JsonHelper.convert(marketOrders));
        }
        return marketOrders;
    }
    /**
     * 统一方法获取uid
     *
     * @param bidRequest
     * @return
     */
    private String getUid(BidRequest bidRequest) {
        Device device = bidRequest.getDevice();
        if (device == null) {
            return null;
        }
        DeviceInfo info = new DeviceInfo()
                .setImei(device.getImei())
                .setImeiMd5(device.getImei_md5())
                .setOaid(device.getOaid())
                .setOaidMd5(device.getOaid_md5())
                .setAnid(device.getDpid())
                .setAnidMd5(device.getDpid_md5())
                .setIdfa(device.getIdfa())
                .setIdfaMd5(device.getIdfa_md5())
                .setOs(device.isAndroid() ? DeviceInfo.OS_ANDROID : device.isIos() ? DeviceInfo.OS_IOS : DeviceInfo.OS_UNKNOWN);
        return DeviceHelper.getUid(info);
    }

    /**
     * 获取请求对应的广告位
     * @param bidRequest
     * @param impression
     * @return
     */
    private Position getPositionByBidRequest(BidRequest bidRequest, Impression impression) {
        String tagId = impression.getTag_id();
        Position position = null;
        if (StringUtils.isBlank(tagId)) {
            //bes广告位映射逻辑
            if (Objects.nonNull(bidRequest.getMediaId())) {
                AdxConstants.Media media = AdxConstants.Media.of(Integer.parseInt(bidRequest.getMediaId()));
                if (AdxConstants.Media.Bes == media) {
                    App app = bidRequest.getApp();
                    position = positionService.getByPositionId(properties.getBesAppId(), impression.getW(),
                            impression.getH(), impression.getImp_type(), app.getId());
                }
            }
        } else {
            position = positionService.getByPositionId(tagId);
        }
        return position;
    }
}
