package com.bxm.adx.common.autoconfigure;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.OpenlogConstants;
import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.cipher.PriceCipher;
import com.bxm.adx.common.entity.CreativeControlDotParam;
import com.bxm.adx.common.entity.DspPriceDotParam;
import com.bxm.adx.common.entity.RiskControlDotParam;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.builder.BuildAttribute;
import com.bxm.adx.common.sell.builder.BuildAttributeStringMacros;
import com.bxm.adx.common.sell.request.*;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.device.SimpleDevice;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author allen
 * @date 2021-06-17
 * @since 1.0
 */
@Configuration
@Slf4j
public class BuildAttributeMacrosAutoConfiguration {
    private final AdxProperties adxProperties;

    public BuildAttributeMacrosAutoConfiguration(AdxProperties adxProperties) {
        this.adxProperties = adxProperties;
    }


    @Bean
    public BuildAttributeStringMacros spm() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.SPM;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return attribute.getSspRequest().getId();
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros ver() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.VER;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return "";
            }
        };
    }


    @Bean
    public BuildAttributeStringMacros bidid() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.BIDID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return attribute.getSspRequest().getId();
            }
        };
    }

    /**
     * 使用缓存的response中的bidid
     *
     * @return
     */
    @Bean
    public BuildAttributeStringMacros bididCache() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.BIDIDC;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                String requestId = attribute.getSspRequest().getId();
                if (Objects.nonNull(attribute.getAdxResponse())) {
                    String responseId = attribute.getAdxResponse().getId();
                    if (!requestId.equalsIgnoreCase(responseId)) {
                        return responseId;
                    } else {
                        return "";
                    }
                }
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros tagId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.TAGID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return attribute.getTagId();
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros dTagId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.DTAGID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return attribute.getdTagId();
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros timestamp() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.TIMESTAMP;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return String.valueOf(System.currentTimeMillis());
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros dAppId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.DAPPID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return attribute.getdAppId();
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros appId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.APPID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return attribute.getAppId();
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros price() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.PRICE;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return attribute.getPrice();
            }

            @Override
            public String getIfBlankValue() {
                return OpenlogConstants.Macros.PRICE;
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros configId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.CONFIGID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return Objects.nonNull(attribute.getConfigId()) ? attribute.getConfigId().toString() : "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros dspId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.DSPID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return Objects.nonNull(attribute.getDspId()) ? attribute.getDspId().toString() : "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros dspPrice() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.DPRICE;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                String dspPrice = Objects.nonNull(attribute.getDspResponsePrice()) ? attribute.getDspResponsePrice().toString() : null;
                if (StringUtils.isNotEmpty(dspPrice)) {
                    try {
                        dspPrice = PriceCipher.encryptWin(dspPrice, adxProperties.getCipherProperties().getDspPriceKey());
                        dspPrice = URLEncoder.encode(dspPrice, "UTF-8");
                    } catch (Exception e) {
                        log.error("cipher err", e);
                    }
                }
                return dspPrice;
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }


    @Bean
    BuildAttributeStringMacros dspWinPrice() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.DSP_WIN_PRICE;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                Bid bid = attribute.getBid();
                if (Objects.nonNull(bid)) {
                    Integer type = bid.getAdv_wpt();
                    if (Objects.nonNull(type) && DispatcherPriceConfig.ADV_WIN_PRICE_TYPE_MEDIA == type) {
                        return null;
                    }
                }

                String dspWinPrice = Objects.nonNull(attribute.getDspResponseWinPrice()) ? attribute.getDspResponseWinPrice().toString() : null;
                dspWinPrice = StringUtils.isEmpty(dspWinPrice) ? Objects.nonNull(attribute.getDspResponsePrice()) ? attribute.getDspResponsePrice().toString() : null : dspWinPrice;
                if (StringUtils.isNotEmpty(dspWinPrice)) {
                    try {
                        dspWinPrice = PriceCipher.encryptWin(dspWinPrice, adxProperties.getCipherProperties().getDspPriceKey());
                        dspWinPrice = URLEncoder.encode(dspWinPrice, "UTF-8");
                    } catch (Exception e) {
                        log.error("cipher err", e);
                    }
                }
                return dspWinPrice;
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros mediaId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.MEDIA_ID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return attribute.getMediaId();
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros chgtypm() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.CHARGE_TYPE_MEDIA;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                Integer chargeType = attribute.getMediaChargeType();
                if (Objects.isNull(chargeType)) {
                    return null;
                }
                return chargeType.toString();
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros chgtypd() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.CHARGE_TYPE_DSP;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                Integer chargeType = attribute.getDspChargeType();
                if (Objects.isNull(chargeType)) {
                    return null;
                }
                return chargeType.toString();
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros device() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.DEVICE;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                Device device = attribute.getSspRequest().getDevice();
                SimpleDevice simpleDevice = SimpleDevice.builder()
                        .imei(device.getImei())
                        .imeiMd5(device.getImei_md5())
                        .androidid(device.getDpid())
                        .androididMd5(device.getDpid_md5())
                        .oaid(device.getOaid())
                        .oaidMd5(device.getOaid_md5())
                        .idfa(device.getIdfa())
                        .idfaMd5(device.getIdfa_md5())
                        .gaid(device.getGaid())
                        .gaidMd5(device.getGaid_md5())
                        .ipv4(device.getIp())
                        .ipv6(device.getIp_v6())
                        .caid(device.getCaidInfoStr())
//                        .caidVersion(device.getCaid_version())
//                        .bootMark(device.getBoot_mark())
//                        .updateMark(device.getUpdate_mark())
                        .build();
                return SimpleDevice.Base64Cipher.encrypt(simpleDevice);
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros encryptionModel() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.ENC_MODEL;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                BidRequest bidRequest = attribute.getSspRequest();
                return Objects.isNull(bidRequest.getEncModel()) ? null : bidRequest.getEncModel().toString();
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros riskControl() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.RISK_CONTROL;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                BidRequest bidRequest = attribute.getSspRequest();
                App app = bidRequest.getApp();
                Device device = bidRequest.getDevice();
                User user = bidRequest.getUser();
                Geo geo = bidRequest.getGeo();
                RiskControlDotParam.RiskControlDotParamBuilder builder = RiskControlDotParam.builder();
                if (Objects.nonNull(app)) {
                    builder.appid(app.getId())
                            .appName(app.getName())
                            .appVer(app.getVer())
                            .appBundle(app.getBundle())
                            .appCat(app.getCat());
                }
                if (Objects.nonNull(device)) {
                    builder.ip(device.getIp())
                            .deviceType(Objects.nonNull(device.getDevice_type()) ? device.getDevice_type().toString() : null)
                            .brand(device.getBrand())
                            .os(device.getOs())
                            .osv(device.getOsv());
                }
                if (Objects.nonNull(user)) {
                    builder.userId(user.getId())
                            .keywords(user.getKeywords());
                }
                if (Objects.nonNull(geo)) {
                    builder.city(geo.getCity());
                }
                String str = JsonHelper.convert(builder.build());
                try {
                    return URLEncoder.encode(str, "UTF-8");
                } catch (UnsupportedEncodingException e) {
                    return null;
                }
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    /**
     * 20230525 应小米ADX要求不能明文传输IP
     *
     * @return
     */
    @Bean
    @Deprecated
    public BuildAttributeStringMacros ip() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.IP;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return attribute.getSspRequest().getDevice().getIp();
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    BuildAttributeStringMacros createId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.CREATE_ID;
            }

            @Override
            public String apply(BuildAttribute buildAttribute) {
                BidResponse response = buildAttribute.getAdxResponse();
                if (response == null || CollectionUtils.isEmpty(response.getSeat_bid())) {
                    return null;
                }
                List<Bid> bid = response.getSeat_bid().get(0).getBid();
                if (CollectionUtils.isEmpty(bid)) {
                    return null;
                }
                return bid.get(0).getAdxCreateId();
            }
        };
    }

    @Bean
    BuildAttributeStringMacros appPackageName() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.APP_PACKAGE_NAME;
            }

            @Override
            public String apply(BuildAttribute buildAttribute) {
                BidRequest sspRequest = buildAttribute.getSspRequest();
                if (sspRequest == null) {
                    return null;
                }
                App app = sspRequest.getApp();
                if (app == null || StringUtils.isEmpty(app.getBundle())) {
                    return null;
                }
                return app.getBundle();
            }
        };
    }

    @Bean
    BuildAttributeStringMacros requestSize() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.REQUEST_SIZE;
            }

            @Override
            public String apply(BuildAttribute buildAttribute) {
                BidRequest sspRequest = buildAttribute.getSspRequest();
                if (sspRequest == null) {
                    return null;
                }
                String size = "";
                Impression impression = sspRequest.getImps().get(0);
                if (impression.getW() != null) {
                    size = size + impression.getW();
                }
                if (impression.getH() != null) {
                    size = size + "*" + impression.getH();
                }
                if (StringUtils.isEmpty(size)) {
                    return null;
                }
                return size;
            }
        };
    }

    @Deprecated
    @Bean
    public BuildAttributeStringMacros dispatcherId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.DISPATCHER_ID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return Objects.isNull(attribute.getDispatcherId()) ? "" : attribute.getDispatcherId().toString();
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros nbr() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.NBR;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                BidResponse dspResponse = attribute.getDspResponse();
                if (Objects.nonNull(dspResponse)) {
                    Integer nbr = dspResponse.getNbr();
                    if (Objects.nonNull(nbr)) {
                        return nbr.toString();
                    }
                }
                return "";
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros strategyId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.STRATEGY_ID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return Objects.isNull(attribute.getStrategyId()) ? "" : attribute.getStrategyId().toString();
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros algoFlowControlBucket() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.ALGO_FC_BUCKET;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                return Objects.isNull(attribute.getAlgoFlowControlBucket()) ? "" : attribute.getAlgoFlowControlBucket();
            }
        };
    }

    /**
     * 此参数仅用于埋点3，4，41中，在对应的事件中单独处理
     *
     * @return
     */
    @Deprecated
    @Bean
    public BuildAttributeStringMacros creatives() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.CREATIVES;
            }

            @Override
            public String apply(BuildAttribute attribute) {
//                if (AdxConstants.BidModel.SUPPORT_CPM == attribute.getSspRequest().getBid_model()) {
//                    return "";
//                }
                return Optional.ofNullable(attribute.getDspResponse() == null ? attribute.getAdxResponse() : attribute.getDspResponse())
                        .map(BidResponse::getSeat_bid)
                        .filter(seatBids -> !CollectionUtils.isEmpty(seatBids))
                        .map(seatBids -> seatBids.stream()
                                .flatMap(seatBid -> seatBid.getBid().stream())
                                .map(bid -> CreativeControlDotParam.builder()
                                        .creativeId(bid.getCreate_id())
                                        .ticketId(bid.getAdid())
                                        .price(bid.getBid())
                                        .dspId(bid.getDspId())
                                        .dappid(bid.getDsp_appid())
                                        .dtagid(bid.getDsp_posid())
                                        .build())
                                .collect(Collectors.toList()))
                        .filter(params -> !CollectionUtils.isEmpty(params))
                        .map(params -> {
                            try {
                                return URLEncoder.encode(JsonHelper.convert(params), "UTF-8");
                            } catch (UnsupportedEncodingException e) {
                                log.error("Creative encode error", e);
                            }
                            return null;
                        })
                        .orElse("");
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros taskId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.TASK_ID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                Bid bid = attribute.getBid();
                if (Objects.nonNull(bid)) {
                    return bid.getTask_id();
                }
                return null;
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros dispatcherPriceConfigId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.DISPATCHER_PC_IC;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                Bid bid = attribute.getBid();
                if (Objects.nonNull(bid)) {
                    return bid.getDpc_id();
                }
                return null;
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros adId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.AD_ID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                Bid bid = attribute.getBid();
                if (Objects.nonNull(bid)) {
                    return bid.getAdid();
                }
                return null;
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros dspWinPriceModel() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.DSP_WIN_PRICE_MODEL;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                Bid bid = attribute.getBid();
                if (Objects.nonNull(bid) && Objects.nonNull(bid.getAdv_wpt())) {
                    return bid.getAdv_wpt().toString();
                }
                return null;
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros dspPriceConfig() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.DSP_PRICE_CONFIG;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                Bid bid = attribute.getBid();
                if (Objects.nonNull(bid)) {
                    Integer advWinPriceType = bid.getAdv_wpt();
                    if (Objects.nonNull(advWinPriceType) && DispatcherPriceConfig.ADV_WIN_PRICE_TYPE_MEDIA == advWinPriceType) {
                        DspPriceDotParam dspPriceDotParam = DspPriceDotParam.builder()
                                .dpr(bid.getDsp_price())
                                .dprpre(bid.getDsp_dis_price())
                                .pm(bid.getAdv_pm())
                                .build();
                        try {
                            String text = PriceCipher.encryptWin(JsonHelper.convert(dspPriceDotParam), adxProperties.getCipherProperties().getDspPriceKey());
                            String encode = URLEncoder.encode(text, "UTF-8");
                            return encode;
                        } catch (Exception e) {
                            log.error("err", e);
                        }
                    }
                }
                return null;
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros usl() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.USL;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                Bid bid = attribute.getBid();
                if (Objects.nonNull(bid)) {
                    if (Objects.nonNull(bid.getUser_score_level())) {
                        return String.valueOf(bid.getUser_score_level());
                    }
                }
                return null;
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros expId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.EXP_ID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                try {
                    BidRequest sspRequest = attribute.getSspRequest();
                    Byte expId = sspRequest.getExpId();
                    return String.valueOf(expId);
                } catch (Exception e) {
                    log.error("expId get exception e:", e);
                    return "";
                }
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }

    @Bean
    public BuildAttributeStringMacros extTagId() {
        return new BuildAttributeStringMacros() {
            @Override
            public String getMacroString() {
                return OpenlogConstants.Macros.EXT_TAG_ID;
            }

            @Override
            public String apply(BuildAttribute attribute) {
                BidRequest sspRequest = attribute.getSspRequest();
                return sspRequest.getImps().iterator().next().getExt_tag_id();
            }

            @Override
            public String getIfBlankValue() {
                return "";
            }
        };
    }
}
