package com.bxm.adx.common.buy.dispatcher.filter;

import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherContext;
import com.bxm.adx.common.buy.dispatcher.abtest.DispatcherABConfig;
import com.bxm.adx.common.buy.dispatcher.abtest.DispatcherABConfigChangeHandler;
import com.bxm.adx.common.buy.dispatcher.abtest.DispatcherConfig;
import com.bxm.adx.common.caching.Id;
import com.bxm.adx.common.ip.IpService;
import com.bxm.adx.common.rule.Rule;
import com.bxm.adx.common.rule.WhiteBlackSetRule;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.common.utils.MapHelper;
import com.bxm.warcar.ip.IP;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.annotation.Configuration;

import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.stream.Collectors;

/**
 * @author fgf
 * @date 2023/3/13
 **/
@DispatcherFilterCondition(probeOn = false)
@Slf4j
@Configuration
public class RegionFilter implements DispatcherFilter<Dispatcher>, DispatcherABConfigChangeHandler {
    private final ConcurrentHashMap<String, Set<String>> dispatcherRegionMap = new ConcurrentHashMap<>();
    private final CopyOnWriteArraySet<String> whiteDispatcherIdSet = Sets.newCopyOnWriteArraySet();
    private final IpService ipService;

    public RegionFilter(IpService ipService) {
        this.ipService = ipService;
    }

    @Override
    public void filter(DispatcherContext<Dispatcher> context, Set<Id> trash) {
        Set<Dispatcher> regionDispatcher = context.getValues().stream().filter(dispatcher -> Objects.nonNull(dispatcher.getCitys())).collect(Collectors.toSet());
        if (CollectionUtils.isEmpty(regionDispatcher)) {
            return;
        }
        IP ip = initIp(context.getRequest());
        Set<Dispatcher> removes = regionDispatcher.stream().filter(
                dispatcher -> limitByRegion(dispatcher, ip)
        ).collect(Collectors.toSet());
        if (CollectionUtils.isNotEmpty(removes)) {
            trash.addAll(removes);
        }
    }

    @Override
    public int getOrder() {
        return 0;
    }

    private boolean limitByRegion(Dispatcher dispatcher, IP ip) {
        String id = dispatcher.getId();
        Set<String> values = dispatcherRegionMap.get(id);
        boolean isWhite = whiteDispatcherIdSet.contains(id);
        if (CollectionUtils.isEmpty(values)) {
            log.warn("Position {} Dispatcher {} region empty", dispatcher.getPositionId(), dispatcher.getId());
            return isWhite;
        }
        if (Objects.isNull(ip)) {
            return isWhite;
        }
        if (ip.isHit()) {
            String hitcode = ip.getHitcode();
            String province = getCodeForProvince(hitcode);
            String city = getCodeForCity(hitcode);
            boolean b = values.contains(hitcode) || values.contains(province) || values.contains(city);
            if (b && !isWhite) {
                return true;
            }
            if (!b && isWhite) {
                return true;
            }
            return false;
        } else {
            return isWhite;
        }
    }

    private String getCodeForProvince(String code) {
        return StringUtils.rightPad(StringUtils.left(code, 2), 6, "0");
    }

    private String getCodeForCity(String code) {
        return StringUtils.rightPad(StringUtils.left(code, 4), 6, "0");
    }

    private IP initIp(BidRequest bidRequest) {
        IP ip = AdxContextFactory.get().getIp();
        if (Objects.nonNull(ip)) {
            return ip;
        }
        Device device = bidRequest.getDevice();
        if (device != null) {
            String ipStr = device.getIp();
            if (!org.springframework.util.StringUtils.isEmpty(ipStr)) {
                ip = ipService.analyze(ipStr);
            }
        }
        if (null == ip) {
            ip = new IP();
        }
        AdxContextFactory.get().setIp(ip);
        return ip;
    }

    @Override
    public void doUpdate(DispatcherABConfig old, DispatcherABConfig latest) {
        if (isExecuteUpdateAndDoDelete(old, latest)) {
            List<DispatcherConfig> dispatcherConfigs = latest.getDispatcherConfigCaches();
            if (CollectionUtils.isEmpty(dispatcherConfigs)) {
                return;
            }
            for (DispatcherConfig dispatcherConfig : dispatcherConfigs) {
                List<Dispatcher> dispatchers = dispatcherConfig.getDispatcherDspCaches();
                if (CollectionUtils.isEmpty(dispatchers)) {
                    continue;
                }
                for (Dispatcher dispatcher : dispatchers) {
                    if (dispatcher.getOpened() == Dispatcher.DISPATCHER_OPENED_NO) {
                        continue;
                    }
                    Rule rule = dispatcher.getCitys();
                    String id = dispatcher.getId();
                    if (Objects.isNull(rule)) {
                        continue;
                    }
                    WhiteBlackSetRule whiteBlackSetRule = new WhiteBlackSetRule(rule);
                    Set<String> values = whiteBlackSetRule.getSet();
                    if (whiteBlackSetRule.isWhite()) {
                        whiteDispatcherIdSet.add(id);
                    }
                    MapHelper.get(dispatcherRegionMap, id, new CopyOnWriteArraySet<>()).addAll(values);
                }
            }
        }
    }

    @Override
    public void doDelete(DispatcherABConfig old) {
        List<DispatcherConfig> dispatcherConfigs = old.getDispatcherConfigCaches();
        if (CollectionUtils.isEmpty(dispatcherConfigs)) {
            return;
        }
        for (DispatcherConfig dispatcherConfig : dispatcherConfigs) {
            List<Dispatcher> dispatchers = dispatcherConfig.getDispatcherDspCaches();
            if (CollectionUtils.isEmpty(dispatchers)) {
                continue;
            }
            for (Dispatcher dispatcher : dispatchers) {
                Rule rule = dispatcher.getCitys();
                if (Objects.isNull(rule)) {
                    continue;
                }
                WhiteBlackSetRule whiteBlackSetRule = new WhiteBlackSetRule(rule);
                Set<String> values = whiteBlackSetRule.getSet();
                String id = dispatcher.getId();
                if (whiteBlackSetRule.isWhite()) {
                    whiteDispatcherIdSet.remove(id);
                }
                dispatcherRegionMap.remove(id);
            }
        }
    }
}
