package com.bxm.adx.common.caching;

import com.bxm.warcar.datasync.client.cache.MutableCache;
import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.jmx.export.annotation.ManagedOperation;
import org.springframework.jmx.export.annotation.ManagedResource;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 抽象的本地缓存
 *
 * @author allen
 * @since 1.0
 * @param <T> 对象模型
 */
@Slf4j
@ManagedResource
public abstract class AbstractLoadingCache<T> implements MutableCache {

    protected final ConcurrentHashMap<String, T> caching = new ConcurrentHashMap<>();

    /**
     * 在更新之后处理
     * @param old 更新前的对象
     * @param newest 更新后的对象
     */
    protected abstract void afterUpdate(T old, T newest);

    /**
     * 在删除之后处理
     * @param old 删除前的对象
     */
    protected abstract void afterDelete(T old);

    @Override
    public T set(String key, Object value) throws NullPointerException {
        Preconditions.checkArgument(StringUtils.isNotBlank(key), "key");
        Preconditions.checkNotNull(value, "value");

        T old = this.get(key);
        try {
            T v = (T) value;
            caching.put(key, v);
            afterUpdate(old, v);
        } catch (Exception e) {
            log.error("set: ", e);
        }
        return old;
    }

    @Override
    public void del(String key) {
        Preconditions.checkArgument(StringUtils.isNotBlank(key), "key");
        try {
            afterDelete(this.get(key));
            caching.remove(key);
        } catch (Exception e) {
            log.error("del: ", e);
        }
    }

    /**
     * 从缓存里获取指定 key 的对象。
     * @param key key
     * @return 对象
     * @throws NullPointerException key是空的
     */
    @Override
    @ManagedOperation
    public T get(String key) throws NullPointerException {
        Preconditions.checkArgument(StringUtils.isNotBlank(key), "key");
        try {
            return caching.get(key);
        } catch (Exception e) {
            log.error("get: ", e);
        }
        return null;
    }

    @Override
    public long size() {
        return caching.size();
    }

    @Override
    public Map<String, Object> getAll() {
        return Collections.unmodifiableMap(caching);
    }

    /**
     * 复制一份内存并返回
     * @return 复制后的集合
     */
    public Map<String, T> duplicateHashMap() {
        return new HashMap<>(caching);
    }
}
