package com.bxm.adx.common.log;

import com.bxm.adx.common.autoconfigure.GlobalServerConfig;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author fgf
 * @date 2024/9/6
 **/
@Slf4j
@Component
public class DailyLogService implements DisposableBean {
    private static final Integer LIMIT = 100;
    private static Cache<String, String> counterCache = CacheBuilder.newBuilder()
            .expireAfterWrite(1, TimeUnit.HOURS)
            .build();
    private final BlockingQueue<AdxDataLog> queue = new LinkedBlockingQueue<>(1000);
    private final ThreadPoolExecutor executor = new ThreadPoolExecutor(1, 1, 0, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(), new NamedThreadFactory("daily-log"));
    private final Counter counter;
    private final ByteLogger byteLogger;
    private final GlobalServerConfig globalServerConfig;
    public DailyLogService(Counter counter, ByteLogger byteLogger, GlobalServerConfig globalServerConfig) {
        this.counter = counter;
        this.byteLogger = byteLogger;
        this.globalServerConfig = globalServerConfig;
        this.startHandleQueue();
    }

    public void log(AdxDataLog dataLog) {
        if (!globalServerConfig.hit()) {
            return;
        }
        // 如果队列满了，直接丢弃了。
        if (!queue.offer(dataLog)) {
            if (log.isDebugEnabled()) {
                log.debug("the queue was full.");
            }
        }
    }

    private boolean limit(AdxDataLog dataLog) {
        String positionId = dataLog.getPositionId();
        if (StringUtils.isEmpty(positionId)) {
            return true;
        }
        int hour = getDateHour();
        String cacheKey = buildLocalCacheKey(positionId, dataLog.getSource(), hour);
        if (null != counterCache.getIfPresent(cacheKey)) {
            return true;
        }
        Long count = counter.incrementAndGet(buildKey(positionId, dataLog.getSource(), hour), 4000);
        if (count > LIMIT) {
            counterCache.put(cacheKey, "");
            return true;
        }
        return false;
    }

    private void startHandleQueue() {
        executor.execute(() -> {
            for (;;) {
                try {
                    AdxDataLog dataLog = queue.take();
                    if (limit(dataLog)) {
                        continue;
                    }
                    byteLogger.add(dataLog);
                } catch (Exception e) {
                    log.warn("log err", e);
                }
            }
        });
    }

    /**
     * 获取当前小时数
     * @return
     */
    private int getDateHour() {
        return LocalDateTime.now().getHour();
    }

    private KeyGenerator buildKey(String positionId, LogSourceEnum logSourceEnum, int hour) {
        return () -> KeyBuilder.build("ADX", "LOG", "REQ", hour, positionId, logSourceEnum.name());
    }

    private String buildLocalCacheKey(String positionId, LogSourceEnum logSourceEnum, int hour) {
        return positionId + "-" + logSourceEnum.name() + "-" + hour;
    }

    @Override
    public void destroy() throws Exception {
        this.executor.shutdownNow();
    }
}
