package com.bxm.adx.common.market.exchange.rebuild.offer;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.market.exchange.rebuild.response.ResponseBuildAttribute;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.util.*;

/**
 * @author fgf
 * @date 2023/1/10
 **/
@Configuration
@Slf4j
public class OfferFactory implements ApplicationListener<ApplicationReadyEvent> {
    private Map<OfferType, Offer> offerMap = new HashMap<>();

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        Collection<Offer> offerList = event.getApplicationContext().getBeansOfType(Offer.class).values();
        for (Offer offer : offerList) {
            offerMap.put(offer.offerType(), offer);
        }
    }

    public Map<Bid, OfferResult> offer(Collection<Bid> bids, ResponseBuildAttribute attribute) {
        Dispatcher dispatcher = attribute.getDispatcher();
        BidRequest request = attribute.getBidRequest();

        Map<Bid, OfferResult> offerResultMap = new HashMap<>(bids.size());
        for (Bid bid : bids) {
            DispatcherPriceConfig priceConfig = attribute.getBidPriceConfig(bid);
            if (Objects.isNull(priceConfig)){
                continue;
            }
            OfferType offerType = getOfferType(dispatcher, priceConfig);
            checkOfferType(offerType, request);
            Offer offer = offerMap.get(offerType);
            if (Objects.isNull(offer)) {
                log.warn("offer is null, dispatcher {}", dispatcher.getId());
                offerResultMap.put(bid, OfferResult.builder()
                        .offer(bid.getPrice())
                        .chargeType(bid.getCharge_type())
                        .budgetType(bid.getBudget_type())
                        .build());
                break;
            }
            offerResultMap.put(bid, offer.offer(bid, request, dispatcher, priceConfig));
        }
        return offerResultMap;
    }

    /**
     * 媒体cpc参竞目前只支持固价/按广告主出价模式
     *
     * @param offerType
     * @param request
     */
    private void checkOfferType(OfferType offerType, BidRequest request) {
        Integer bidModel = Optional.ofNullable(request.getBid_model()).orElse(AdxConstants.BidModel.SUPPORT_CPM);
        if (AdxConstants.BidModel.SUPPORT_CPC == bidModel) {
            if (OfferType.Dsp_Avg_Price != offerType && OfferType.Bid_By_Dsp != offerType) {
                log.error("Media bid-model is cpc, Only supported type Dsp_Avg_Price/Bid_By_Dsp");
                throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
            }
        }

        if (AdxConstants.BidModel.SUPPORT_CPM == bidModel) {
            if (OfferType.Bid_By_Dsp == offerType) {
//                log.error("Media bid-model is cpm,  unsupported type Bid_By_Dsp");
                throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
            }
        }

        if (OfferType.Dsp_Avg_Price_Discount == offerType) {
            log.error("Warning Type Dsp_Avg_Price_Discount");
            throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
        }
    }

    /**
     * 从不同版本的配置中获取媒体出价方式
     *
     * @param dispatcher
     * @return
     */
    private OfferType getOfferType(Dispatcher dispatcher, DispatcherPriceConfig dispatcherPriceConfig) {
        //优先使用分时段设置里的配置，外部配置仍然保留，仅用于防错
        Integer mediaOfferType = Objects.isNull(dispatcherPriceConfig) ? null : dispatcherPriceConfig.getMediaOfferType();
        if (Objects.nonNull(mediaOfferType)) {
            switch (mediaOfferType) {
                case AdxConstants.DispatcherAvgType.FIXED:
                    return OfferType.Dsp_Avg_Price;
                case AdxConstants.DispatcherAvgType.NONE:
                    return OfferType.NONE_Dsp_Avg_Price;
                case AdxConstants.DispatcherAvgType.FIXED_DISCOUNT:
                    return OfferType.Dsp_Avg_Price_Discount;
                case AdxConstants.DispatcherAvgType.BID_BY_DSP:
                    return OfferType.Bid_By_Dsp;
            }
        }
        log.warn("position {}, dispatcher {}, mediaOfferType is null", dispatcher.getPositionId(), dispatcher.getId());
        Integer avgType = dispatcher.getAvgType();
        if (Objects.isNull(avgType)) {
            //适配旧版本 since 20230522
            BigDecimal avg = dispatcher.getDspAvgPrice();
            if (Objects.isNull(avg)) {
                return OfferType.NONE_Dsp_Avg_Price;
            }
            return OfferType.Dsp_Avg_Price;
        } else {
            switch (avgType) {
                case AdxConstants.DispatcherAvgType.FIXED:
                    return OfferType.Dsp_Avg_Price;
                case AdxConstants.DispatcherAvgType.NONE:
                    return OfferType.NONE_Dsp_Avg_Price;
                case AdxConstants.DispatcherAvgType.FIXED_DISCOUNT:
                    return OfferType.Dsp_Avg_Price_Discount;
                default:
                    log.error("unknown dispatcher {} avg-type {}", dispatcher.getId(), dispatcher.getAvgType());
            }
            throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
        }
    }
}
