package com.bxm.adx.common.market.exchange.rebuild.response;

import cn.hutool.core.collection.CollUtil;
import com.alibaba.fastjson.JSON;
import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.request.Impression;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.warcar.integration.pair.Pair;
import com.google.common.collect.Lists;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

/**
 * 检查Bid，填充必要/默认参数
 *
 * @author fgf
 * @date 2023/1/9
 **/
@Slf4j
@Configuration
public class BidBuilder implements AdxBidResponseBuilder {
    private final Pair pair;
    private final static String KEY = "adx.deeplink.bundle.mapping";
    // 限制生效的媒体 id key
    private final static String MEDIA_KEY ="adx.deeplink.bundle.mapping.media";


    public BidBuilder(Pair pair) {
        this.pair = pair;
    }

    @Override
    public int getOrder() {
        return 2;
    }

    @Override
    public void rebuildAdxBidResponse(BidResponse bidResponse, ResponseBuildAttribute attribute) {
        Dispatcher dispatcher = attribute.getDispatcher();
        BidRequest adxRequest = attribute.getAdxRequest();
        Impression impression = adxRequest.getImps().iterator().next();
        for (SeatBid seatBid : bidResponse.getSeat_bid()) {
            for (Bid bid : seatBid.getBid()) {
                //设置bid#id
                if (StringUtils.isEmpty(bid.getId())) {
                    bid.setId(RandomStringUtils.randomAlphabetic(8));
                }
                //设置adid
                if (StringUtils.isEmpty(bid.getAdid())) {
                    bid.setAdid(RandomStringUtils.randomAlphanumeric(5));
                }
                //默认tagid为媒体广告位id
                if (StringUtils.isEmpty(bid.getTag_id())) {
                    bid.setTag_id(impression.getTag_id());
                }
                //
                if (StringUtils.isEmpty(bid.getCreate_id())) {
                    bid.setCreate_id(RandomStringUtils.randomNumeric(7));
                }
                Integer chargeType = Optional.ofNullable(bid.getCharge_type()).orElse(AdxConstants.ChargeType.CPM);
                bid.setCharge_type(chargeType);
                bid.setDsp_charge_type(chargeType);
                Integer budgetType = Optional.ofNullable(bid.getBudget_type()).orElse(AdxConstants.ChargeType.CPM);
                bid.setBudget_type(budgetType);
                bid.setDsp_budget_type(budgetType);
                //记录dsp出价
                switch (budgetType) {
                    case AdxConstants.ChargeType.CPM:
                        bid.setDsp_price(new BigDecimal(bid.getPrice().toString()));
                        break;
                    case AdxConstants.ChargeType.CPC:
                        bid.setDsp_price(new BigDecimal(bid.getBid().toString()));
                        break;
                }

                bid.setDsp_appid(dispatcher.getDspAppid());
                bid.setDsp_posid(dispatcher.getDspPosid());
                bid.setDspId(dispatcher.getDspId());
                if (Objects.isNull(bid.getImp_id())) {
                    bid.setImp_id(impression.getId());
                }
                if (Objects.isNull(bid.getType())) {
                    Integer type = Optional.ofNullable(AdxConstants.Type.getTypeByImpType(impression.getImp_type()))
                            .orElse(AdxConstants.Type.IMAGE.getType());
                    bid.setType(type);
                }
                //设置bid对应的价格配置，注意此类的顺序
                String dpcId = bid.getDpc_id();
                if (StringUtils.isNotEmpty(dpcId)) {
                    Optional<DispatcherPriceConfig> optional = dispatcher.getDispatcherPriceConfigs().stream().filter(priceConfig ->
                            priceConfig.getPriceModeId().toString().equals(dpcId)).findFirst();
                    if (optional.isPresent()) {
                        bid.setTrans_type(optional.get().getConversionTarget());
                        attribute.putBidPriceConfig(bid, optional.get());
                    } else {
                        log.warn("dispatcher {} dpcid {} config is null", dispatcher.getId(), dpcId);
                    }
                }

                //dp广告填充包名
                try {
                    String url = StringUtils.isNotBlank(bid.getUniversal_link()) ? bid.getUniversal_link() : bid.getDeep_link_url();
                    if (StringUtils.isBlank(url)) {
                        continue;
                    }
                    Integer cType = bid.getC_type();
                    List<Integer> dpCtypes = Lists.newArrayList(AdxConstants.Action.DEEP_LINK.getAction(), AdxConstants.Action.DEEP_LINK_DOWNLOAD.getAction());
                    if(Objects.isNull(cType) || !CollUtil.contains(dpCtypes, cType)){
                        continue;
                    }
                    String mediaId = adxRequest.getMediaId();
                    Set<String> mediaIds = pair.get(MEDIA_KEY).ofHashSet();
                    if (CollUtil.isEmpty(mediaIds) || mediaIds.contains(mediaId)) {
                     String mapping = pair.get(KEY).of();
                     if (StringUtils.isNotBlank(mapping)){
                         List<AppInfo> appInfos = JSON.parseArray(mapping, AppInfo.class);
                        if (CollUtil.isEmpty(appInfos)){
                            continue;
                        }
                        for (AppInfo appInfo : appInfos) {
                            if (StringUtils.isNotBlank(appInfo.getDeepLinkUrl()) && StringUtils.startsWith(url, appInfo.getDeepLinkUrl())) {
                                if (StringUtils.isBlank(bid.getBundle())) {
                                     bid.setBundle(appInfo.getBundle());
                                }
                                if (StringUtils.isBlank(bid.getApp_name())){
                                    bid.setApp_name(appInfo.getName());
                                }
                                break;
                                }
                            }
                        }
                    }
                } catch (Exception e) {
                    log.error("bidBuilder setBundle error", e);
                }
            }
        }
    }
    @Data
    private static class AppInfo {
        private String deepLinkUrl;
        private String bundle;
        private String name;
        public AppInfo() {
        }
        public AppInfo(String deepLinkUrl, String bundle, String name) {
            this.deepLinkUrl = deepLinkUrl;
            this.bundle = bundle;
            this.name = name;
        }
    }
}
