package com.bxm.adx.common.market.filter;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.mcssp.common.enums.app.DockingMethodTypeEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 底价过滤
 *
 * @author fgf
 * @date 2023/3/21
 **/
@Slf4j
@Configuration
@DealFilterCondition(preview = true)
public class PriceBidFloorFilter implements Filter {
    @Override
    public void filter(List<Deal> deals, Set<Deal> trash) {
        Set<Deal> remove = deals.stream().filter(
                deal -> {
                    BidResponse response = deal.getBidResponse();
                    BidRequest request = deal.getBidRequest();
                    List<SeatBid> seatBids = response.getSeat_bid();
                    seatBids.removeIf(seatBid -> {
                        List<Bid> bidList = seatBid.getBid();
                        bidList.removeIf(bid -> {
                            boolean isCpc = isCpc(bid);
                            BigDecimal price = null;
                            int chargeType = bid.getCharge_type();
                            switch (chargeType) {
                                case AdxConstants.ChargeType.CPM:
                                    price = bid.getPrice();
                                    break;
                                case AdxConstants.ChargeType.CPC:
                                    price = bid.getBid();
                                    break;
                                case AdxConstants.ChargeType.OCPM:
                                    price = bid.getObid();
                                    break;
                                default:
                                    break;
                            }
                            if (Objects.isNull(price) || price.compareTo(BigDecimal.ZERO) == 0) {
                                return true;
                            }
                            //api接入模式下需要进行底价过滤 ocpm 不进行底价过滤
                            BigDecimal bidFloor = null;
                            if (isApi(request)  && !isOcpm(bid)) {
                                if (!isCpc) {
                                    bidFloor = request.getImps().iterator().next().getBid_floor();
                                } else {
                                    bidFloor = request.getImps().iterator().next().getCpc_bid_floor();
                                }
                            }
                            if (Objects.nonNull(bidFloor)) {
                                if (price.compareTo(bidFloor) < 0) {
                                    return true;
                                }
                            }
                            return false;
                        });
                        return CollectionUtils.isEmpty(bidList);
                    });
                    return CollectionUtils.isEmpty(seatBids);
                }
        ).collect(Collectors.toSet());
        if (CollectionUtils.isNotEmpty(remove)) {
            trash.addAll(remove);
        }
    }

    @Override
    public int getOrder() {
        return 1;
    }

    private boolean isApi(BidRequest request) {
        if (Objects.nonNull(request.getDockingMethodType())
                && DockingMethodTypeEnum.API_OPERATION.getType().equals(request.getDockingMethodType())) {
            return true;
        }
        return false;
    }

    private boolean isCpc(Bid bid) {
        int chargeType = bid.getCharge_type();
        return AdxConstants.ChargeType.CPC == chargeType;
    }

    private boolean isOcpm(Bid bid) {
        int chargeType = bid.getCharge_type();
        return AdxConstants.ChargeType.OCPM == chargeType;
    }
}
