package com.bxm.adx.common.market.filter;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.adx.common.utils.MapHelper;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 低价过滤器，同一个广告位只保留出价最高的一个CPM。
 * 只支持单个seatbid
 *
 * @author allen
 * @since 2019-12-18
 */
@Slf4j
@Configuration
public class PriceLowerFilter implements Filter {

    @Override
    public void filter(List<Deal> deals, Set<Deal> trash) {
        BidRequest request = AdxContextFactory.get().getBidRequest();
        int bidModel = request.getBid_model();
        //获取最高cpm最高出价
        Optional<Bid> topPriceBid = getTopPriceBid(deals);
        if (!topPriceBid.isPresent()) {
            if (AdxConstants.BidModel.SUPPORT_CPM == bidModel) {
                if (log.isDebugEnabled()) {
                    log.debug("No valid price");
                }
                //CPM模式下，找不到正常出价，排除所有deal
                deals.removeIf(deal -> {
                    return true;
                });
            }
            return;
        }
        final BigDecimal topPrice = topPriceBid.get().getPrice();

        Map<String, BigDecimal> higher = Maps.newHashMap();
        String key = "higher";
        Set<Deal> remove = deals.stream().filter(
                deal -> {
                    BidResponse response = deal.getBidResponse();

                    List<SeatBid> seatBids = response.getSeat_bid();
                    seatBids.removeIf(seatBid -> {
                        List<Bid> bidList = seatBid.getBid();

                        bidList.removeIf(b -> {
                            int chargeType = b.getCharge_type();
                            if (AdxConstants.ChargeType.CPM != chargeType) {
                                return false;
                            }

                            BigDecimal price = b.getPrice();

                            //比价逻辑: 比最高价格小的直接过滤;和最高价相同的取第一个取到的bid，其他都过滤;比最高价高的输出告警
                            if (price.compareTo(topPrice) < 0) {
                                return true;
                            } else if (price.compareTo(topPrice) == 0) {
                                if (org.springframework.util.CollectionUtils.isEmpty(higher)) {
                                    MapHelper.get(higher, key, price);
                                    return false;
                                } else {
                                    return true;
                                }
                            } else {
                                log.warn("warning bid top price wrong, top price = {},  deal = {}", topPrice.toString(), JsonHelper.convert(deal));
                                MapHelper.get(higher, key, price);
                                return false;
                            }
                        });
                        return CollectionUtils.isEmpty(bidList);
                    });
                    return CollectionUtils.isEmpty(seatBids);
                }
        ).collect(Collectors.toSet());
        if (CollectionUtils.isNotEmpty(remove)) {
            trash.addAll(remove);
        }
    }

    @Override
    public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE;
    }


    /**
     * 获取最高价bid
     * @param deals
     * @return
     */
    private Optional<Bid> getTopPriceBid(List<Deal> deals) {
        return deals.stream()
                .filter(deal -> Objects.nonNull(deal.getBidResponse()))
                .map(Deal::getBidResponse)
                .filter(bidResponse -> CollectionUtils.isNotEmpty(bidResponse.getSeat_bid()))
                .map(BidResponse::getSeat_bid)
                .flatMap(Collection::stream)
                .filter(seatBid -> CollectionUtils.isNotEmpty(seatBid.getBid()))
                .map(SeatBid::getBid)
                .flatMap(Collection::stream)
                .filter(bid -> Objects.nonNull(bid.getPrice()) && Objects.nonNull(bid.getCharge_type()) &&
                        AdxConstants.ChargeType.CPM == bid.getCharge_type())
                //按照价格从大到小排列取第一个
                .sorted(Comparator.comparing(Bid::getPrice).reversed()).findFirst();
    }
}
