package com.bxm.adx.common.micrometer;

import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.warcar.utils.MapHelper;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.binder.MeterBinder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.annotation.Configuration;

import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author fgf
 * @date 2022/12/12
 */
@Slf4j
@Configuration
public class SellerMeter implements MeterBinder {
    /**
     * 流量方填充广告返回统计
     */
    private final ConcurrentHashMap<String, Counter> paddedCounter = new ConcurrentHashMap<>();
    /**
     * 流量方未命中广告位统计
     */
    private final ConcurrentHashMap<String, Counter> positionMissCounter = new ConcurrentHashMap<>();
    private MeterRegistry registry;

    @Override
    public void bindTo(MeterRegistry meterRegistry) {
        this.registry = meterRegistry;
    }

    /**
     * 统计不同异常
     *
     * @param e
     * @param path
     */
    public void increaseException(Exception e, String path) {
        if (!(e instanceof AdxException)) {
            return;
        }
        AdxErrEnum errEnum = ((AdxException) e).getAdxErrEnum();
        if (Objects.isNull(errEnum)) {
            return;
        }
        switch (errEnum) {
            case POSITION_NOT_FOUND:
                increasePositionMiss(path);
                break;
        }
        return;
    }

    /**
     * 统计流量方广告填充情况
     *
     * @param path
     */
    public void increasePadding(String path) {
        if (StringUtils.isEmpty(path)) {
            return;
        }
        Counter counter = MapHelper.get(paddedCounter, path, new MapHelper.InitializingValue<Counter>() {
            @Override
            public Counter initializing() {
                return Counter.builder("adx.requests.padding").tags("path", path).register(registry);
            }
        });
        if (Objects.nonNull(counter)) {
            counter.increment();
        }
    }

    /**
     * 统计流量方广告位命中情况
     *
     * @param path
     */
    public void increasePositionMiss(String path) {

        if (StringUtils.isEmpty(path)) {
            return;
        }
        Counter counter = MapHelper.get(positionMissCounter, path, new MapHelper.InitializingValue<Counter>() {
            @Override
            public Counter initializing() {
                return Counter.builder("adx.requests.miss").tags("path", path).register(registry);
            }
        });
        if (Objects.nonNull(counter)) {
            counter.increment();
        }
    }
}
