package com.bxm.adx.common.openlog.listener.internal;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.OpenlogConstants;
import com.bxm.adx.common.market.exchange.rebuild.filter.FilterInfo;
import com.bxm.adx.common.openlog.event.internal.AdxDispatcherPriceConfigFilterEvent;
import com.bxm.adx.common.sell.builder.BuildAttribute;
import com.bxm.adx.common.sell.builder.BuildAttributeStringMacrosHandler;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.openlog.extension.client.OpenLogClient;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import lombok.Data;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author fgf
 * @date 2024/6/11
 **/
@Slf4j
@Configuration
public class DotAdxDispatcherPriceConfigFilterEventListener implements EventListener<AdxDispatcherPriceConfigFilterEvent> {
    private OpenLogClient openLogClient;
    private final AdxProperties properties;
    private final BuildAttributeStringMacrosHandler macrosHandler;

    public DotAdxDispatcherPriceConfigFilterEventListener(AdxProperties properties, BuildAttributeStringMacrosHandler macrosHandler) {
        this.properties = properties;
        this.macrosHandler = macrosHandler;
    }

    @Autowired
    public void setOpenLogClient(OpenLogClient openLogClient) {
        this.openLogClient = openLogClient;
    }

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(AdxDispatcherPriceConfigFilterEvent event) {
        if (Math.abs(event.getBidRequest().getId().hashCode() % 100) >= properties.getDotSimplingPercent()) {
            return;
        }
        MultiValueMap<String, String> customizeParams = buildCustomizeParams(event.getFilterInfos());
        String url = properties.getOpenLog().create(true, OpenlogConstants.Mt.DISPATCHER_PRICE_CONFIG_FILTER, customizeParams, OpenlogConstants.WITHOUT_MACROS_INNER);
        BuildAttribute attribute = new BuildAttribute(OpenlogConstants.Mt.DSP_RESPONSE_MT)
                .setDspId(event.getBuyer().getDsp().getId())
                .setSspRequest(event.getBidRequest())
                .setAdxRequest(event.getAdxRequest())
                .setConfigId(event.getDispatcher().getConfigId())
                .setStrategyId(event.getDispatcher().getStrategyId());
        url = macrosHandler.replaceAll(url, attribute);
        if (log.isDebugEnabled()) {
            log.debug("dot mt {} url {}", OpenlogConstants.Mt.DISPATCHER_PRICE_CONFIG_FILTER, url);
        }
        openLogClient.asyncRequest(url);
    }

    private MultiValueMap<String, String> buildCustomizeParams(List<FilterInfo> filterInfos) {
        boolean multiple = filterInfos.size() > 1;
        List<FilterEntity> entities = new ArrayList<>();
        for (FilterInfo filterInfo : filterInfos) {
            Bid bid = filterInfo.getBid();
            String adId = bid.getAdid();
            Map<Integer, Set<Long>> filterConfigMap = filterInfo.getFilterConfigMap();
            filterConfigMap.forEach(
                    (integer, longs) -> {
                        FilterEntity filter = new FilterEntity();
                        filter.setIds(longs).setType(integer);
                        if (multiple) {
                            filter.setAdid(adId);
                        }
                        entities.add(filter);
                    }
            );
        }
        MultiValueMap<String, String> customizeParams = new LinkedMultiValueMap<>();
        customizeParams.add(OpenlogConstants.MacrosCustomizeParams.DISPATCHER_PRICE_CONFIG_FILTER, UrlHelper.urlEncode(JsonHelper.convert(entities)));
        return customizeParams;
    }

    @Data
    @Accessors(chain = true)
    static class FilterEntity {
        private String adid;
        private Integer type;
        private Set<Long> ids;
    }
}
