package com.bxm.adx.common.sell.position.cached;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.position.PositionChangeHandler;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.annotation.Configuration;

import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author fgf
 * @date 2022/12/14
 **/
@Configuration
@Slf4j
public class PositionCached implements PositionChangeHandler {
    private final AdxProperties properties;
    /**
     * 广告位ID-广告位集合
     * <pre>
     * |-------------|-----------------------------|
     * |    Key      | Value                       |
     * |-------------|-----------------------------|
     * | {positionId}|[position,position,...]      |
     * |-------------|-----------------------------|
     * </pre>
     */
    private final ConcurrentHashMap<String, Position> positionIdMap = new ConcurrentHashMap<>();
    /**
     * 媒体广告位ID-广告位集合
     * <pre>
     * |--------------------|-----------------------------|
     * |    Key             | Value                       |
     * |--------------------|-----------------------------|
     * | {mediaPositionId}  |[position,position,...]      |
     * |--------------------|-----------------------------|
     * </pre>
     */
    private final ConcurrentHashMap<String, Position> mediaPositionIdMap = new ConcurrentHashMap<>();

    public PositionCached(AdxProperties properties) {
        this.properties = properties;
    }

    @Override
    public void doUpdate(Position old, Position latest) {
        if (isExecuteUpdateAndDoDelete(old, latest)) {

            positionIdMap.put(latest.getPositionId(), latest);

            String mediaPositionId = latest.getAppPositionId();
            if (StringUtils.isNotEmpty(mediaPositionId)) {
                for (String tmp : mediaPositionId.split(",")) {
                    if (StringUtils.isNotEmpty(tmp)) {
                        mediaPositionIdMap.put(tmp, latest);
                    }
                }
            }
        }
    }

    @Override
    public void doDelete(Position old) {
        positionIdMap.remove(old.getPositionId());
        String mediaPositionId = old.getAppPositionId();
        if (StringUtils.isNotEmpty(mediaPositionId)) {
            for (String tmp : mediaPositionId.split(",")) {
                if (StringUtils.isNotEmpty(tmp)) {
                    mediaPositionIdMap.remove(tmp);
                }
            }
        }
    }

    /**
     * 根据bxm广告位id/媒体广告位id获取广告位
     *
     * @param tagId
     * @return
     */
    public Position getPositionByTagId(String tagId) {
        return Optional.ofNullable(positionIdMap.get(tagId)).orElse(mediaPositionIdMap.get(tagId));
    }

    /**
     * 根据bxm广告位id获取广告位
     *
     * @param positionId
     * @return
     */
    public Position getPositionById(String positionId) {
        return positionIdMap.get(positionId);
    }

    /**
     * 根据媒体广告位ID获取广告位
     *
     * @param positionId
     * @return
     */
    public Position getPositionByAppTagId(String positionId) {
        return mediaPositionIdMap.get(positionId);
    }

    /**
     * 是否是bes广告位
     *
     * @param position
     * @return
     */
    @Deprecated
    private boolean isBesPosition(Position position) {
        if (!org.springframework.util.StringUtils.isEmpty(position.getPositionSize())) {
            if (position.getAppId().equals(properties.getBesAndroidAppId())) {
                return true;
            }
            if (position.getAppId().equals(properties.getBesIosAppId())) {
                return true;
            }
        }
        return false;
    }
}
