package com.bxm.adx.common.buy.dispatcher;

import com.bxm.adx.common.CacheKeys;
import com.bxm.adx.common.DispatcherFlowProperties;
import com.bxm.adx.common.buy.dispatcher.abtest.DispatcherABConfig;
import com.bxm.adx.common.buy.dispatcher.abtest.DispatcherABConfigChangeHandler;
import com.bxm.adx.common.buy.dispatcher.abtest.DispatcherConfig;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.request.App;
import com.bxm.adx.common.utils.DateUtils;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.xcache.Fetcher;
import com.bxm.warcar.xcache.TargetFactory;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.RandomUtils;
import org.springframework.scheduling.annotation.Scheduled;
import redis.clients.jedis.*;

import java.time.LocalDate;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CopyOnWriteArraySet;

/**
 * 砍量业务处理
 *
 * @author fgf
 * @date 2023/5/4
 **/
@Slf4j
public class DefaultDispatcherFlowControl implements DispatcherABConfigChangeHandler, DispatcherFlowControl {

    private final long SLEEP_TIME = 3000;
    private final CopyOnWriteArraySet<String> flowControlDispatchers = new CopyOnWriteArraySet<>();

    private final JedisPool jedisPool;
    private final Fetcher fetcher;
    private final DispatcherFlowProperties properties;

    public DefaultDispatcherFlowControl(DispatcherFlowProperties properties, JedisPool jedisPool, Fetcher fetcher) {
        this.jedisPool = jedisPool;
        this.properties = properties;
        this.fetcher = fetcher;
    }

    @Override
    public boolean flowControl(Dispatcher dispatcher, BidRequest request) {
        String date = LocalDate.now().format(DateUtils.FORMAT_SIMPLE);
        KeyGenerator key = getKeyGenerator(date, dispatcher, request);
        Integer percent = getPercent(key);
        if (Objects.nonNull(percent)) {
            percent = fixPercent(key, percent);
            int random = RandomUtils.nextInt(0, 100);
            if (random < percent) {
                return true;
            }
        }
        return false;
    }

    /**
     * 获取砍量百分比
     * @param key
     * @return
     */
    public Integer getPercent(KeyGenerator key) {
        return fetcher.fetch(new TargetFactory<Integer>()
                .keyGenerator(key)
                .cls(Integer.class)
                .selector(properties.getDb())
                .skipNativeCache(properties.getSkipNativeCache())
                .build());
    }

    private KeyGenerator getKeyGenerator(String date, Dispatcher dispatcher, BidRequest request) {
        String dspId = dispatcher.getDspId().toString();
        String dspTagId = dispatcher.getDspPosid();
        String dspAppId = dispatcher.getDspAppid();
        String tagId = request.getImps().iterator().next().getTag_id();
        App app = request.getApp();
        String appPkg = "";
        if (Objects.nonNull(app)) {
            appPkg = Optional.ofNullable(app.getBundle()).orElse("");
        }

        return CacheKeys.DispatcherFlow.getDispatcherFlowPercent(date, dspId, dspAppId, dspTagId, tagId, appPkg);
    }

    /**
     * 修正百分比
     * @param keyGenerator
     * @param percent
     * @return
     */
    private Integer fixPercent(KeyGenerator keyGenerator, Integer percent) {
        if (percent.compareTo(DispatcherFlowProperties.CEILING) > 0) {
            log.warn("key {}, val {}", keyGenerator.generateKey(), percent);
            return DispatcherFlowProperties.CEILING;
        }
        if (percent.compareTo(DispatcherFlowProperties.FLOOR) < 0) {
            log.warn("key {}, val {}", keyGenerator.generateKey(), percent);
            return DispatcherFlowProperties.FLOOR;
        }
        return percent;
    }

    @Scheduled(cron = "0 */10 * * * ?")
    public void execute() {
        long start = System.currentTimeMillis();
        String date = LocalDate.now().format(DateUtils.FORMAT_SIMPLE);
        for (String partKey : flowControlDispatchers) {
            try (Jedis jedis = jedisPool.getResource()) {
                jedis.select(properties.getDb());
                // 设置Scan参数
                String previewKey = CacheKeys.DispatcherFlow.baseFlow(date);
                ScanParams params = new ScanParams()
                        .match(KeyBuilder.build(previewKey, partKey, "*"))
                        .count(properties.getPipelineCount());
                // 初始化游标
                String cursor = "0";
                do {
                    // 执行Scan命令
                    ScanResult<String> result = jedis.scan(cursor, params);
                    // 获取新的游标
                    cursor = result.getStringCursor();
                    try (Pipeline pipeline = jedis.pipelined()) {
                        // 获取Scan命令返回的是否砍量key列表
                        List<String> keys = result.getResult();
                        // 获取数据计算的是否砍量
                        List<Object> flowControlList = pipelineGetResponse(pipeline, keys);
                        // 获取砍量百分比key
                        List<String> percentKeys = getPercentKeys(keys);
                        // 获取砍量百分比
                        List<Object> percentList = pipelineGetResponse(pipeline, percentKeys);
                        //重新设置百分比
                        List<Object> cutResponse = cutting(pipeline, flowControlList, percentKeys, percentList);
                        if (log.isDebugEnabled()) {
                            log.debug("pipeline keys {}, response {}", percentKeys, cutResponse);
                        }
                    }
                } while (!cursor.equals("0"));
            } catch (Exception e) {
                log.error("err", e);
            }
        }
        //控制任务执行时间，降低因为系统（系统时间误差）等原因造成分布式锁失效，多台服务器都执行定时任务的概率
        long end = System.currentTimeMillis();
        if (end - start < SLEEP_TIME) {
            try {
                Thread.sleep(SLEEP_TIME);
            } catch (InterruptedException e) {
            }
        }
    }

    /**
     * 操作砍量百分比
     *
     * @param pipeline
     * @param flowControlList
     * @param percentKeys
     * @param percentList
     * @return
     */
    private List<Object> cutting(Pipeline pipeline, List<Object> flowControlList, List<String> percentKeys, List<Object> percentList) {
        Long cr = properties.getCr();
        Integer expire = properties.getExpire();
        for (int i = 0; i < flowControlList.size(); i++) {
            Object cut = flowControlList.get(i).toString();
            String percentKey = percentKeys.get(i);
            Object old = percentList.get(i);
            if (DispatcherFlowProperties.YES.equals(String.valueOf(cut))) {
                if (!outOfBounds(true, old)) {
                    pipeline.incrBy(percentKey, cr);
                }
            } else {
                if (!outOfBounds(false, old)) {
                    pipeline.decrBy(percentKey, cr);
                }
            }
            pipeline.expire(percentKey, expire);
        }
        List<Object> percentResponse = pipeline.syncAndReturnAll();
        return percentResponse;
    }

    /**
     * 砍量百分比是否越界
     *
     * @param isCeiling
     * @param old
     * @return
     */
    private boolean outOfBounds(boolean isCeiling, Object old) {
        try {
            if (isCeiling) {
                if (Objects.isNull(old)) {
                    return false;
                } else {
                    Integer oldVal = Integer.valueOf(String.valueOf(old));
                    if (oldVal.compareTo(DispatcherFlowProperties.CEILING) >= 0) {
                        return true;
                    } else {
                        return false;
                    }
                }
            } else {
                if (Objects.isNull(old)) {
                    return true;
                } else {
                    Integer oldVal = Integer.valueOf(String.valueOf(old));
                    if (oldVal.compareTo(DispatcherFlowProperties.FLOOR) > 0) {
                        return false;
                    } else {
                        return true;
                    }
                }
            }
        } catch (Exception e) {
            log.error("bounds err", e);
            return false;
        }
    }

    /**
     * 执行get方法的pipeline
     *
     * @param pipeline
     * @param keys
     * @return
     */
    private List<Object> pipelineGetResponse(Pipeline pipeline, List<String> keys) {
        for (String key : keys) {
            pipeline.get(key);
        }
        return pipeline.syncAndReturnAll();
    }

    /**
     * 获取存放砍量百分比的key
     *
     * @param keys
     * @return
     */
    private List<String> getPercentKeys(List<String> keys) {
        List<String> percentKeys = Lists.newArrayListWithCapacity(keys.size());
        for (String key : keys) {
            percentKeys.add(percentKey(key));
        }
        return percentKeys;
    }

    private String percentKey(String key) {
        String base = CacheKeys.DispatcherFlow.baseFlow("");
        String[] params = key.split(base);
        return CacheKeys.DispatcherFlow.basePercent(params[1]);
    }

    @Override
    public void doUpdate(DispatcherABConfig old, DispatcherABConfig latest) {
        if (isExecuteUpdateAndDoDelete(old, latest)) {
            List<DispatcherConfig> dispatcherConfigs = latest.getDispatcherConfigCaches();
            if (CollectionUtils.isEmpty(dispatcherConfigs)) {
                return;
            }

            for (DispatcherConfig dispatcherConfig : dispatcherConfigs) {
                List<Dispatcher> dispatchers = dispatcherConfig.getDispatcherDspCaches();
                if (CollectionUtils.isEmpty(dispatchers)) {
                    continue;
                }
                for (Dispatcher dispatcher : dispatchers) {
                    if (dispatcher.isOpened()) {
                        Byte flowSwitch = dispatcher.getChopQuantitySwitch();
                        if (Objects.nonNull(flowSwitch) && Dispatcher.FLOW_OPENED_YES == flowSwitch) {
                            flowControlDispatchers.add(getPartKey(dispatcher));
                        }
                    }
                }
            }
        }
    }

    @Override
    public void doDelete(DispatcherABConfig old) {
        List<DispatcherConfig> dispatcherConfigs = old.getDispatcherConfigCaches();
        if (CollectionUtils.isEmpty(dispatcherConfigs)) {
            return;
        }

        for (DispatcherConfig dispatcherConfig : dispatcherConfigs) {
            List<Dispatcher> dispatchers = dispatcherConfig.getDispatcherDspCaches();
            if (CollectionUtils.isEmpty(dispatchers)) {
                continue;
            }
            for (Dispatcher dispatcher : dispatchers) {
                flowControlDispatchers.remove(getPartKey(dispatcher));
            }
        }
    }

    private String getPartKey(Dispatcher dispatcher) {
        return KeyBuilder.build(dispatcher.getDspId(), dispatcher.getDspAppid(), dispatcher.getDspPosid(), dispatcher.getPositionId());
    }
}
