package com.bxm.adx.common.caching.sync;

import com.bxm.warcar.datasync.client.cache.MutableCache;
import com.bxm.warcar.zk.listener.children.ChildrenChanged;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;

import java.util.Map;
import java.util.Objects;

/**
 * 数据同步 mutable 接口实现，收到数据变更后按照对象类型分发给匹配的 {@link DataSyncHandler} 实现。
 *
 * @author allen
 * @date 2021-06-15
 * @since 1.0
 * @see DataSyncHandler
 */
@Slf4j
@Primary
@Configuration
public class DataSyncMutableCacheDispatcher implements MutableCache, ApplicationListener<ApplicationReadyEvent> {

    private final Map<Class<?>, DataSyncHandler> handlers = Maps.newHashMap();

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        //PathChildrenCache.startMode = NORMAL, 默认加载子节点下所有数据，并监听新的事件
        ChildrenChanged childrenChanged = event.getApplicationContext().getBean(ChildrenChanged.class);
        childrenChanged.init();

        Map<String, DataSyncHandler> beans = event.getApplicationContext().getBeansOfType(DataSyncHandler.class);
        for (Map.Entry<String, DataSyncHandler> entry : beans.entrySet()) {
            DataSyncHandler handler = entry.getValue();
            handlers.put(handler.getClazz(), handler);
            log.info("Registered {}", handler);
        }
    }

    @Override
    public synchronized Object set(String key, Object value) throws NullPointerException {

        // 使用同步的原因：
        // 由于 MapHelper 非线程安全的，而内部缓存处理器有大量使用，所以暂时在这里控制同步。
        // by Allen
        if (Objects.isNull(key)) {
            log.warn("key is null!");
            return null;
        }
        if (Objects.isNull(value)) {
            log.warn("value is null by key: {}", key);
            return null;
        }
        Class<?> clazz = value.getClass();
        DataSyncHandler dataSyncHandler = handlers.get(clazz);
        if (Objects.isNull(dataSyncHandler)) {
            log.warn("Does not found DataSyncHandler matching class {}", clazz);
            return null;
        }
        return dataSyncHandler.set(key, value);
    }

    @Override
    public void del(String key) {
    }

    @Override
    public <T> T get(String key) throws NullPointerException {
        throw new UnsupportedOperationException();
    }

    @Override
    public Map<String, Object> getAll() {
        throw new UnsupportedOperationException();
    }

    @Override
    public long size() {
        return -1;
    }
}
