package com.bxm.adx.common.limiter;

import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.utils.NamedThreadFactory;
import lombok.extern.slf4j.Slf4j;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author fgf
 * @date 2023/4/4
 **/
@Slf4j
public class DefaultInsertExpireHandler implements InsertExpireHandler {
    private final BlockingQueue<DefaultInsertExpireHandler.QueueEntity> queue = new LinkedBlockingQueue<>(10000);
    private final ThreadPoolExecutor executor = new ThreadPoolExecutor(1, 1, 0, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(), new NamedThreadFactory("insert-expire"));
    private final Updater updater;
    private final Fetcher fetcher;
    public DefaultInsertExpireHandler(Updater updater, Fetcher fetcher) {
        this.updater = updater;
        this.fetcher = fetcher;
        this.startHandleQueue();
    }

    @Override
    public void insert(KeyGenerator hashKey, String field, String val, int expire) {
        // 如果队列满了，直接丢弃了。
        if (!queue.offer(new QueueEntity(hashKey, field, val, expire))) {
            if (log.isDebugEnabled()) {
                log.debug("the queue was full.");
            }
        }
    }


    private void startHandleQueue() {
        executor.execute(() -> {
            for (;;) {
                try {
                    QueueEntity entity = queue.take();
                    KeyGenerator key = entity.hashKey;
                    int ttl = fetcher.ttl(key).intValue();
                    int expire = entity.expire;
                    if (log.isDebugEnabled()) {
                        log.debug("ttl {}, expire {}.", ttl, expire);
                    }
                    //如果key原有的ttl比此次设置的expire大，则仍使用原ttl
                    if (ttl > entity.expire) {
                        expire = ttl;
                    }
                    updater.hupdate(key, entity.field, entity.val, expire);
                    if (log.isDebugEnabled()) {
                        log.debug("update key hash {} val {}.", key.generateKey(), entity.field);
                    }
                } catch (Exception e) {
                    log.info("take: {}", e.getMessage());
                }
            }
        });
    }

    private static class QueueEntity {

        KeyGenerator hashKey;
        String field;
        String val;
        int expire;

        private QueueEntity(KeyGenerator hashKey, String field, String val, int expire) {
            this.hashKey = hashKey;
            this.field = field;
            this.val = val;
            this.expire = expire;
        }
    }

    interface Ttl {
        int KEY_NOT_EXIST = -2;
        int KET_NO_TTL = -1;
    }
}
