package com.bxm.adx.common.market.exchange.rebuild.response;

import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.context.annotation.Configuration;

import java.util.*;

/**
 * 填充dispatcher里面的具体出价配置
 * 从多个出价策略挑选命中的，如果没命中使用默认出价策略里的时段配置
 *
 * @author lijingyu
 * @date 2023/1/9
 **/
@Slf4j
//@Configuration
public class PriceConfigResponseBuilder implements AdxBidResponseBuilder {

    @Override
    public int getOrder() {
        return 3;
    }

    @Override
    public void rebuildAdxBidResponse(BidResponse bidResponse, ResponseBuildAttribute attribute) {
        //新逻辑会设置价格配置Map，跳过这个类原有的逻辑
        if (MapUtils.isNotEmpty(attribute.getBidPriceConfigMap())) {
            return;
        }
        Dispatcher dispatcher = attribute.getDispatcher();
        List<DispatcherPriceConfig> dispatcherPriceConfigs = dispatcher.getDispatcherPriceConfigs();
        if (CollectionUtils.isEmpty(dispatcherPriceConfigs)) {
            log.warn("position {}, dispatcher {}, price config is empty", dispatcher.getPositionId(), dispatcher.getId());
            return;
        }
        try {
            for (SeatBid seatBid : bidResponse.getSeat_bid()) {
                for (Bid bid : seatBid.getBid()) {
                    //根据taskId匹配白名单策略，匹配不到就是走默认策略
                    String taskId = bid.getTask_id();
                    //用户质量分
                    Integer userScoreLevel = bid.getUser_score_level();
                    if (log.isDebugEnabled()) {
                        log.debug(" adid:{},taskId:{},userScoreLevel:{}", bid.getAdid(), taskId,userScoreLevel);
                    }

                    //获取任务id/用户质量分定向的价格配置
                    Optional<DispatcherPriceConfig> optional = dispatcherPriceConfigs.stream()
                            .filter(config -> (config.getFilterMode() != null && DispatcherPriceConfig.FILTER_MODEL_WHITE == config.getFilterMode()))
                            .filter(config -> getPriceConfig(config,taskId,userScoreLevel))
                            .findFirst();

                    //如果没有定向的价格配置 则 获取默认策略的价格配置
                    DispatcherPriceConfig selectedConfig = optional.orElseGet(() -> dispatcherPriceConfigs.stream()
                            .filter(config -> config.getFilterMode() == null || DispatcherPriceConfig.FILTER_MODEL_UNLIMITED == config.getFilterMode())
                            .findFirst()
                            .orElse(null));

                    attribute.putBidPriceConfig(bid, selectedConfig);
                    Long priceModeId = selectedConfig.getPriceModeId();
                    bid.setDpc_id(priceModeId == null ? null : priceModeId.toString());
                }
            }
        } catch (Exception e) {
            log.error("position {} dispatcher {} price config {}", dispatcher.getPositionId(), dispatcher.getId(), JsonHelper.convert(dispatcherPriceConfigs));
            throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
        }
    }

    /**
     * 是否命中当前出价
     *
     * @param config 出价配置
     * @param taskId 任务id
     * @param userScoreLevel 用户质量分
     * @return true、false
     */
    private boolean getPriceConfig(DispatcherPriceConfig config, String taskId, Integer userScoreLevel) {
        List<String> taskIds = config.getTaskIds();
        if (CollectionUtils.isEmpty(taskIds)) {
            return false;
        }
        String interval = locateInterval(userScoreLevel);
        return taskIds.contains(taskId) || taskIds.contains(interval) || taskIds.contains(taskId + ":" + interval);
    }

    /**
     * 根据分数转换区间
     * @param number 分数
     * @return 区间
     */
    public String locateInterval(Integer number) {
        if (number == null) {
            return "";
        }

        if (number >= 81 && number <= 100) {
            return "81_100";
        } else if (number >= 61 && number <= 80) {
            return "61_80";
        } else if (number >= 41 && number <= 60) {
            return "41_60";
        } else if (number >= 21 && number <= 40) {
            return "21_40";
        } else if (number >= 1 && number <= 20) {
            return "1_20";
        } else {
            return String.valueOf(number);
        }
    }
}
