package com.bxm.adx.common.market.monitor;

import com.bxm.adx.common.*;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.builder.BuildAttribute;
import com.bxm.adx.common.sell.builder.BuildAttributeStringMacrosHandler;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.common.sell.response.*;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.integration.pair.Pair;
import com.bxm.warcar.utils.UrlHelper;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponentsBuilder;

import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import static com.bxm.adx.common.OpenlogConstants.WITHOUT_MACROS_OUT;

/**
 * 添加bxm-adx监测链接
 *
 * @author fgf
 * @date 2023/4/14
 **/
@Slf4j
@Component
public class AdxMonitorHandler implements MonitorHandler {
    private final String DSP_IMP = "u1";
    private final int[] DP_MT = new int[]{OpenlogConstants.Mt.DP_START_MT, OpenlogConstants.Mt.DP_SUCCESS_MT, OpenlogConstants.Mt.DP_FAIL_MT};
    private final int[] DOWNLOAD_MT = new int[]{OpenlogConstants.Mt.DOWN_FINISH, OpenlogConstants.Mt.DOWN_START, OpenlogConstants.Mt.INSTALL_FINISH, OpenlogConstants.Mt.INSTALL_START};
    private final static String WIN_NOTICE_MEDIA_LIST_KEY = "adx.win.notice.media";

    private final Pair pair;
    private final AdxProperties adxProperties;
    private final BuildAttributeStringMacrosHandler macrosHandler;
    private final Updater updater;

    public AdxMonitorHandler(Pair pair, AdxProperties adxProperties, BuildAttributeStringMacrosHandler macrosHandler, Updater updater) {
        this.pair = pair;
        this.adxProperties = adxProperties;
        this.macrosHandler = macrosHandler;
        this.updater = updater;
    }

    @Override
    public void handler(BuildAttribute attribute) {
        Bid bid = attribute.getBid();
        OpenLogProperties properties = adxProperties.getOpenLog();
        //曝光
        bid.setImp_monitors(impMonitors(bid, attribute, properties));
        //点击
        bid.setClick_monitors(clickMonitors(bid, attribute, properties));
        //deeplink
        if (isDp(bid)) {
            bid.setDpMonitor(dpMonitor(bid, attribute, properties));
        }
        //下载
        if (isDownload(bid)) {
            bid.setApp_monitor(appMonitor(bid, attribute, properties));
        }
        //竞价失败
        if (needFailUrlByMedia(attribute.getMediaId())) {
            String failUrl = adxProperties.getOpenLog().createNeed(false, OpenlogConstants.Mt.BID_FAIL, OpenlogConstants.WITH_MACROS);
            bid.setFail_url(macrosHandler.replaceAll(failUrl, attribute.setMt(OpenlogConstants.Mt.BID_FAIL)));
        }
        //指定媒体支持竞胜埋点
        Set<String> mediaIdList = pair.get(WIN_NOTICE_MEDIA_LIST_KEY).ofHashSet();
        if (!CollectionUtils.isEmpty(mediaIdList) && mediaIdList.contains(attribute.getMediaId())) {
            bid.setNurl(getNurl(bid, attribute, properties));
        }
    }

    @Override
    public int getOrder() {
        return 1;
    }

    /**
     * 获取bxm-adx的正常竞胜监测
     *
     * @param bid
     * @param attribute
     * @param properties
     * @return
     */
    private String getNurl(Bid bid, BuildAttribute attribute, OpenLogProperties properties) {
        //曝光
        int mt = OpenlogConstants.Mt.WIN_MT;
        String nurl = getUrl(properties, mt, "down_x", "down_y", "up_x", "up_y", "ad_width", "ad_height");
        nurl = macrosHandler.replaceAll(nurl, attribute.setMt(mt));
        nurl = replaceMacroPrice(nurl, bid, attribute.getSspRequest());
        return nurl;
    }

    private List<ImpMonitor> impMonitors(Bid bid, BuildAttribute attribute, OpenLogProperties properties) {
        String impUrl = getImpUrl(bid, attribute, properties);
        List<ImpMonitor> impMonitors = bid.getImp_monitors();
        if (imp302(attribute)) {
            if (!CollectionUtils.isEmpty(impMonitors) && impMonitors.size() > 0) {
                String dspUrl = impMonitors.get(0).getImp_monitor_url();
                impMonitors.remove(0);
                String impUrl302 = getImpUrl302(impUrl, dspUrl, attribute);
                impMonitors.add(ImpMonitor.builder().imp_monitor_url(impUrl302).build());
                return impMonitors;
            }
        }
        impMonitors.add(ImpMonitor.builder().imp_monitor_url(impUrl).build());
        return impMonitors;
    }

    /**
     * 获取bxm-adx的正常曝光监测
     *
     * @param bid
     * @param attribute
     * @param properties
     * @return
     */
    private String getImpUrl(Bid bid, BuildAttribute attribute, OpenLogProperties properties) {
        //曝光
        int mt = OpenlogConstants.Mt.IMP_MT;
        String impUrl = getImpUrl(properties, mt, buildImpCustomizeParams(bid, attribute.getSspRequest()), "down_x", "down_y", "up_x", "up_y", "ad_width", "ad_height");
        impUrl = macrosHandler.replaceAll(impUrl, attribute.setMt(mt));
        impUrl = replaceMacroPrice(impUrl, bid, attribute.getSspRequest());
        return impUrl;
    }

    /**
     * 构建客制化埋点参数
     */
    private MultiValueMap<String, String> buildImpCustomizeParams(Bid bid, BidRequest sspRequest) {
        MultiValueMap<String, String> customizeParams = new LinkedMultiValueMap<>();
        //添加利润率优先实验id
        if (Objects.nonNull(bid.getDpc_strategy_id())){
            customizeParams.add(OpenlogConstants.MacrosCustomizeParams.DPC_STRATEGY_ID, UrlHelper.urlEncode(bid.getDpc_strategy_id()));
        }
        Device device = sspRequest.getDevice();
        if (Objects.nonNull(device)) {
            String verCodeOfHms = device.getVer_code_of_hms();
            if (StringUtils.isNotBlank(verCodeOfHms)) {
                customizeParams.add(OpenlogConstants.MacrosCustomizeParams.HMS_VER, verCodeOfHms);
            }
            String clientTime = device.getClient_time();
            if (StringUtils.isNotBlank(clientTime)) {
                long timestamp = 0;
                try {
                    DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSSZ");
                    ZonedDateTime zonedDateTime = ZonedDateTime.parse(clientTime, formatter);
                    // 转换为时间戳（毫秒值）
                    timestamp = zonedDateTime.toInstant().toEpochMilli();
                } catch (Exception e) {
                    log.error("client_time parse error", e);
                }
                customizeParams.add(OpenlogConstants.MacrosCustomizeParams.CLIENT_TIME, String.valueOf(timestamp));
            }
        }
        return customizeParams;
    }

    /**
     * 获取需要302的曝光
     * 生成新的曝光监测地址，参数中包含的dsp的曝光用于302跳转，缓存bxm-adx的曝光，在收到曝光后服务端上报
     *
     * @param bxmImpUrl
     * @param dspImpUrl
     * @param attribute
     * @return
     */
    private String getImpUrl302(String bxmImpUrl, String dspImpUrl, BuildAttribute attribute) {
        String url = adxProperties.getAdsCounter().getAdx302Url(false, true);
        String replacedUrl = macrosHandler.replaceAll(url, attribute);

        UriComponentsBuilder builder = UriComponentsBuilder.fromUriString(replacedUrl);
        builder.replaceQueryParam(DSP_IMP, UrlHelper.urlEncode(dspImpUrl));

        saveImpUrl(attribute, bxmImpUrl);
        return builder.build().toString();
    }

    private void saveImpUrl(BuildAttribute attribute, String url) {
        KeyGenerator key = CacheKeys.getImpUrl302(
                attribute.getSspRequest().getId(),
                attribute.getMediaId(),
                attribute.getTagId(),
                attribute.getDspId().toString(),
                attribute.getdTagId()
        );
        updater.updateWithSelector(key, url, 24 * 3600, 4);
    }

    private boolean imp302(BuildAttribute attribute) {
        String mediaId = attribute.getMediaId();
        String dspId = attribute.getDspId().toString();
        String appId = attribute.getAppId();
        Imp302Properties properties = adxProperties.getImp302();
        for (Imp302Properties.MediaConfig config : properties.getMediaConfigs()) {
            if (mediaId.equals(config.getMediaId())) {
                List<String> appIds = config.getAppIds();
                List<String> dspIds = config.getDspIds();
                if (!CollectionUtils.isEmpty(appIds)) {
                    if (!appIds.contains(appId)) {
                        continue;
                    }
                }
                if (!CollectionUtils.isEmpty(dspIds)) {
                    if (dspIds.contains(dspId)) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    private List<ClickMonitor> clickMonitors(Bid bid, BuildAttribute attribute, OpenLogProperties properties) {
        //点击
        int mt = OpenlogConstants.Mt.CLICK_MT;
        List<ClickMonitor> clickMonitors = Optional.ofNullable(bid.getClick_monitors()).orElse(Lists.newArrayList());
        String clickUrl = getClickUrl(properties, mt, buildCustomizeParams(bid));
        clickMonitors.add(ClickMonitor.builder().click_monitor_url(macrosHandler.replaceAll(clickUrl, attribute.setMt(mt))).build());
        return clickMonitors;
    }

    private String getClickUrl(OpenLogProperties properties, int mt, MultiValueMap<String, String> customizeParams, String... removeParams) {
        return properties.create(mt, customizeParams , removeParams);
    }

    private String getImpUrl(OpenLogProperties properties, int mt, MultiValueMap<String, String> customizeParams, String... removeParams) {
        return properties.create(mt, customizeParams , removeParams);
    }


    /**
     * 构建客制化埋点参数
     */
    private MultiValueMap<String, String> buildCustomizeParams(Bid bid) {
        MultiValueMap<String, String> customizeParams = new LinkedMultiValueMap<>();
        if (Objects.nonNull(bid.getTrans_type())){
            customizeParams.add(OpenlogConstants.MacrosCustomizeParams.TRANS_TYPE, bid.getTrans_type().toString());
        }
        if (Objects.nonNull(bid.getDpc_strategy_id())){
            customizeParams.add(OpenlogConstants.MacrosCustomizeParams.DPC_STRATEGY_ID, UrlHelper.urlEncode(bid.getDpc_strategy_id()));
        }
        return customizeParams;
    }

    /**
     * deeplink监测
     *
     * @param bid
     * @param attribute
     */
    private DpMonitor dpMonitor(Bid bid, BuildAttribute attribute, OpenLogProperties properties) {
        DpMonitor dpMonitor = Optional.ofNullable(bid.getDpMonitor()).orElse(DpMonitor.builder().build());
        attribute.setMt(null);

        for (int mt : DP_MT) {
            String url = getUrl(properties, mt, WITHOUT_MACROS_OUT);
            url = macrosHandler.replaceAll(url, attribute);
            switch (mt) {
                case OpenlogConstants.Mt.DP_START_MT:
                    dpMonitor.setAwk_start_urls(mergeUrl(dpMonitor.getAwk_start_urls(), url));
                    break;
                case OpenlogConstants.Mt.DP_SUCCESS_MT:
                    dpMonitor.setAwk_success_urls(mergeUrl(dpMonitor.getAwk_success_urls(), url));
                    break;
                case OpenlogConstants.Mt.DP_FAIL_MT:
                    dpMonitor.setAwk_fail_urls(mergeUrl(dpMonitor.getAwk_fail_urls(), url));
                    break;
            }
        }

        return dpMonitor;
    }

    /**
     * 下载监测
     *
     * @param bid
     * @param attribute
     */
    private AppMonitor appMonitor(Bid bid, BuildAttribute attribute, OpenLogProperties properties) {
        AppMonitor appMonitor = Optional.ofNullable(bid.getApp_monitor()).orElse(AppMonitor.builder().build());
        attribute.setMt(null);

        for (int mt : DOWNLOAD_MT) {
            String url = adxProperties.getOpenLog().create(mt, WITHOUT_MACROS_OUT);
            macrosHandler.replaceAll(url, attribute);
            switch (mt) {
                case OpenlogConstants.Mt.DOWN_START:
                    appMonitor.setDs_urls(mergeUrl(appMonitor.getDs_urls(), url));
                    break;
                case OpenlogConstants.Mt.DOWN_FINISH:
                    appMonitor.setDf_urls(mergeUrl(appMonitor.getDf_urls(), url));
                    break;
                case OpenlogConstants.Mt.INSTALL_START:
                    appMonitor.setSs_urls(mergeUrl(appMonitor.getSs_urls(), url));
                    break;
                case OpenlogConstants.Mt.INSTALL_FINISH:
                    appMonitor.setSf_urls(mergeUrl(appMonitor.getSf_urls(), url));
                    break;
            }
        }

        return appMonitor;
    }

    /**
     * 生成bxm-adx监测链接
     *
     * @param properties
     * @param mt
     * @param removeParams
     * @return
     */
    private String getUrl(OpenLogProperties properties, int mt, String... removeParams) {
        return properties.create(mt, removeParams);
    }

    /**
     * 替换价格
     *
     * @param url
     * @param bid
     * @param request
     * @return
     */
    // FIXME: 2023/4/17 应该移到宏参替换中
    private String replaceMacroPrice(String url, Bid bid, BidRequest request) {
        if (AdxConstants.needReplacePrice(request.getMediaId())) {
            String price = "0";
            if (Objects.nonNull(bid.getPrice())) {
                price = bid.getPrice().toString();
            }
            url = url.replace(OpenlogConstants.Macros.PRICE, price);
        }
        return url;
    }

    /**
     * 是否是deeplink广告
     *
     * @param bid
     * @return
     */
    private boolean isDp(Bid bid) {
        return StringUtils.isNotEmpty(bid.getDeep_link_url()) ? true : StringUtils.isNotEmpty(bid.getUniversal_link());
    }

    /**
     * 是否是下载类型广告
     *
     * @param bid
     * @return
     */
    private boolean isDownload(Bid bid) {
        if (Objects.nonNull(bid.getDownload_info())) {
            return true;
        }
        if (StringUtils.isNotEmpty(bid.getApp_download_url())) {
            return true;
        }
        return false;
    }

    /**
     * 合并url到url集合中
     *
     * @param list
     * @param url
     * @return
     */
    private List<String> mergeUrl(List<String> list, String url) {
        list = Optional.ofNullable(list).orElse(Lists.newArrayList());
        list.add(url);
        return list;
    }

    /**
     * 支持失败上报的媒体
     * @param mediaId
     * @return
     */
    private boolean needFailUrlByMedia(String mediaId)  {
        if (StringUtils.isNotEmpty(mediaId)) {
            AdxConstants.Media media = AdxConstants.Media.of(Integer.parseInt(mediaId));
            if (AdxConstants.Media.Oppo == media || AdxConstants.Media.Sigmob == media) {
                return true;
            }
        }
        return false;
    }
}
