package com.bxm.adx.common.openlog.listener.internal;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.OpenlogConstants;
import com.bxm.adx.common.openlog.event.internal.MediaRequestEvent;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.builder.BuildAttribute;
import com.bxm.adx.common.sell.builder.BuildAttributeStringMacrosHandler;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.common.sell.request.Geo;
import com.bxm.openlog.extension.client.OpenLogClient;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.UrlHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Objects;

/**
 * @author fgf
 * @date 2022/6/1
 **/
@Slf4j
@Configuration
public class DotMediaRequestEventListener implements EventListener<MediaRequestEvent> {
    private OpenLogClient openLogClient;
    private final AdxProperties properties;
    private final BuildAttributeStringMacrosHandler macrosHandler;
    private final double EPSILON = 0.01D;

    public DotMediaRequestEventListener(AdxProperties properties, BuildAttributeStringMacrosHandler macrosHandler) {
        this.properties = properties;
        this.macrosHandler = macrosHandler;
    }

    @Autowired
    public void setOpenLogClient(OpenLogClient openLogClient) {
        this.openLogClient = openLogClient;
    }

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(MediaRequestEvent event) {
        if (Math.abs(event.getSspRequest().getId().hashCode() % 100) >= properties.getDotSimplingPercent()) {
            return;
        }
        String url = properties.getOpenLog().create(true, OpenlogConstants.Mt.MEDIA_REQUEST_MT, customizeParams(event), "dspid", "chgtypm", "chgtypd");
        BuildAttribute attribute = new BuildAttribute(OpenlogConstants.Mt.MEDIA_REQUEST_MT).setSspRequest(event.getSspRequest());
        url = macrosHandler.replaceAll(url, attribute);
        openLogClient.asyncRequest(url);
    }

    private MultiValueMap<String, String> customizeParams(MediaRequestEvent event) {
        MultiValueMap<String, String> customizeParams = new LinkedMultiValueMap<>();

        BidRequest request = event.getSspRequest();
        Geo geo = request.getGeo();
        if (Objects.nonNull(geo)) {
            Double lat = geo.getLat();
            Double lon = geo.getLon();
            if (Objects.nonNull(lat) && Objects.nonNull(lon)) {
                if (isZero(lat) || isZero(lon)) {
                    return null;
                }
                customizeParams.add("geo", StringUtils.join(new String[]{String.format("%.2f", lat), String.format("%.2f", lon)}, ","));
            }
        }
        Device device = request.getDevice();
        if (Objects.nonNull(device)) {
            String ua = device.getUa();
            if (StringUtils.isNotBlank(ua)) {
                customizeParams.add("ua", UrlHelper.urlEncode(ua));
            }
            String verCodeOfHms = device.getVer_code_of_hms();
            if (StringUtils.isNotBlank(verCodeOfHms)) {
                customizeParams.add(OpenlogConstants.MacrosCustomizeParams.HMS_VER, verCodeOfHms);
            }
            String clientTime = device.getClient_time();
            if (StringUtils.isNotBlank(clientTime)) {
                long timestamp = 0;
                try {
                    DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSSZ");
                    ZonedDateTime zonedDateTime = ZonedDateTime.parse(clientTime, formatter);
                    // 转换为时间戳（毫秒值）
                    timestamp = zonedDateTime.toInstant().toEpochMilli();
                } catch (Exception e) {
                    log.error("client_time parse error", e);
                }
                customizeParams.add(OpenlogConstants.MacrosCustomizeParams.CLIENT_TIME, String.valueOf(timestamp));
            }
        }

//        App app = request.getApp();
//        if (Objects.nonNull(app)) {
//            String name = app.getName();
//            if (StringUtils.isNotBlank(name)) {
//                customizeParams.add("appname", name);
//            }
//        }
        return customizeParams;
    }

    private boolean isZero(double value) {
        return Math.abs(value) < EPSILON;
    }
}
