package com.bxm.adx.common.sell;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.adapter.BidModelAdapter;
import com.bxm.adx.common.buy.buyers.BuyerWrapper;
import com.bxm.adx.common.buy.buyers.PriorityBuyers;
import com.bxm.adx.common.market.MarketOrders;
import com.bxm.adx.common.market.MarketRequest;
import com.bxm.adx.common.openlog.event.internal.MediaRequestEvent;
import com.bxm.adx.common.sell.init.BidRequestInitializer;
import com.bxm.adx.common.sell.init.InitializerParam;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.position.mapping.PositionMappingStrategyFactory;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.common.sell.request.Impression;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.user.id.generator.DeviceHelper;
import com.bxm.user.id.generator.DeviceInfo;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;

/**
 * @author allen
 * @since 2019-12-20
 */
@Slf4j
@Configuration
public class DefaultBidRequestWrapper implements BidRequestWrapper, ApplicationListener<ApplicationReadyEvent> {

    private final PriorityBuyers priorityBuyers;
    private final AdxProperties properties;
    private final PositionMappingStrategyFactory positionMappingStrategyFactory;
    private final EventPark eventPark;
    private List<BidRequestInitializer> initializers = Lists.newArrayList();

    public DefaultBidRequestWrapper(PriorityBuyers priorityBuyers, AdxProperties properties, PositionMappingStrategyFactory positionMappingStrategyFactory, EventPark eventPark) {
        this.priorityBuyers = priorityBuyers;
        this.properties = properties;
        this.positionMappingStrategyFactory = positionMappingStrategyFactory;
        this.eventPark = eventPark;
    }

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        initializers.addAll(event.getApplicationContext().getBeansOfType(BidRequestInitializer.class).values());
        initializers.sort(Comparator.comparingInt(Ordered::getOrder));
    }

    @Override
    public MarketOrders packing(BidModelAdapter adapter, BidRequest bidRequest) {
        List<Impression> imps = bidRequest.getImps();
        if (CollectionUtils.isEmpty(imps)) {
            throw new AdxException(AdxErrEnum.POSITION_NOT_FOUND);
        }

        Impression impression = bidRequest.getImps().iterator().next();
        //映射广告位
        Position mediaPosition = positionMappingStrategyFactory.getPosition(bidRequest, impression);

        if (Objects.isNull(mediaPosition)) {
            throw new AdxException(AdxErrEnum.POSITION_NOT_FOUND);
        }
        AdxContextFactory.get().setBidRequest(bidRequest);
        String uid = getUid(bidRequest);
        bidRequest = adapter.reBuildBidRequest(bidRequest, mediaPosition);
        if (bidRequest == null) {
            return null;
        }

        Map<Position, List<List<BuyerWrapper>>> positionListMap = Maps.newHashMap();
        List<Position> positionList = new ArrayList<>();

        if (Objects.nonNull(mediaPosition) && mediaPosition.isEnabled()) {
            positionList.add(mediaPosition);
            AdxContextFactory.get().setPosition(mediaPosition);
            //检查填充请求需要的默认参数
            BidRequest finalBidRequest = bidRequest;
            initializers.forEach(init -> init.accept(finalBidRequest, InitializerParam.builder().position(mediaPosition).build()));
            //埋点媒体请求
            eventPark.post(new MediaRequestEvent(this, bidRequest, mediaPosition));
            // 查找这些广告位分配的卖家和等级
            List<List<BuyerWrapper>> buyers =
                    priorityBuyers.findAsPriority(mediaPosition, bidRequest);
            priorityBuyers.rebuildBuyers(bidRequest, mediaPosition, buyers);
            positionListMap.put(mediaPosition, buyers);
        }

        if (CollectionUtils.isEmpty(positionListMap)) {
            throw new AdxException(AdxErrEnum.POSITION_NOT_FOUND);
        }

        // 索引即表示优先级，从0开始。
        List<MarketRequest> requests = Lists.newArrayListWithCapacity(properties.getMaxCapacityOfBuyers());
        // 重组，分出每一个优先级中对应的DSP+广告位
        for (Map.Entry<Position, List<List<BuyerWrapper>>> positionListEntry : positionListMap.entrySet()) {
            Position position = positionListEntry.getKey();
            List<List<BuyerWrapper>> list = positionListEntry.getValue();
            int level = list.size();
            if (level == 0) {
                // 没有设置分流策略
                continue;
            }

            for (int i = 0; i < level; i++) {
                // 这个广告位第i级的DSP配置列表
                List<BuyerWrapper> array = list.get(i);
                if (CollectionUtils.isEmpty(array)) {
                    continue;
                }
                MarketRequest marketRequest = CollectionUtils.isEmpty(requests) ? null : requests.size() > i ? requests.get(i) : null;
                if (null == marketRequest) {
                    marketRequest = new MarketRequest(i);
                    requests.add(i, marketRequest);
                }
                marketRequest.addBuyers(array);
                marketRequest.addPosition(position);
            }
        }

        MarketOrders marketOrders = new MarketOrders(bidRequest, requests, positionList, uid);
        if (log.isDebugEnabled()) {
            log.debug("{}", JsonHelper.convert(marketOrders));
        }
        return marketOrders;
    }

    /**
     * 统一方法获取uid
     *
     * @param bidRequest
     * @return
     */
    private String getUid(BidRequest bidRequest) {
        Device device = bidRequest.getDevice();
        if (device == null) {
            return null;
        }
        DeviceInfo info = new DeviceInfo()
                .setImei(device.getImei())
                .setImeiMd5(device.getImei_md5())
                .setOaid(device.getOaid())
                .setOaidMd5(device.getOaid_md5())
                .setAnid(device.getDpid())
                .setAnidMd5(device.getDpid_md5())
                .setIdfa(device.getIdfa())
                .setIdfaMd5(device.getIdfa_md5())
                .setGaid(device.getGaid())
                .setGaidMd5(device.getGaid_md5())
                .setOs(device.isAndroid() ? DeviceInfo.OS_ANDROID : device.isIos() ? DeviceInfo.OS_IOS : DeviceInfo.OS_UNKNOWN);
        String uid = DeviceHelper.getUid(info);
        if (StringUtils.isEmpty(AdxContextFactory.get().getUid())) {
            AdxContextFactory.get().setUid(uid);
        }
        return uid;
    }
}
