package com.bxm.adx.common.sell.position.mapping;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.PositionMappingProperties;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.request.Impression;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * @author fgf
 * @date 2023/2/22
 **/
@Component
@Slf4j
public class PositionMappingStrategyFactory implements ApplicationListener<ApplicationReadyEvent> {
    private final AdxProperties adxProperties;
    private final Map<Strategy, PositionMappingStrategy> strategyMap = Maps.newHashMap();

    public PositionMappingStrategyFactory(AdxProperties adxProperties) {
        this.adxProperties = adxProperties;
    }

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        Collection<PositionMappingStrategy> strategies = event.getApplicationContext().getBeansOfType(PositionMappingStrategy.class).values();
        if (CollectionUtils.isEmpty(strategies)) {
            log.warn("position mapping strategy is empty");
            return;
        }
        for (PositionMappingStrategy strategy : strategies) {
            strategyMap.put(strategy.strategy(), strategy);
        }
    }

    public Position getPosition(BidRequest bidRequest, Impression impression) {
        PositionMappingProperties properties = getProperties(bidRequest, impression);
        Strategy strategy = Strategy.Default;
        if (Objects.nonNull(properties)) {
            strategy = Optional.ofNullable(properties.getStrategy()).orElse(Strategy.Default);
        }
        return strategyMap.get(strategy).mapping(bidRequest, impression, properties);
    }

    private PositionMappingProperties getProperties(BidRequest bidRequest, Impression impression) {
        String mediaId = bidRequest.getMediaId();

        if (StringUtils.isEmpty(mediaId)) {
            return null;
        }

        List<PositionMappingProperties> properties = adxProperties.getPositionMapping();
        Optional<PositionMappingProperties> optional = findMediaConfig(properties, mediaId);
        return optional.orElse(null);
    }

    private Optional<PositionMappingProperties> findMediaConfig(List<PositionMappingProperties> list, String mediaId) {
        if (CollectionUtils.isEmpty(list)) {
            return Optional.empty();
        }
        return list.stream().filter(mediaConfig -> mediaId.equals(mediaConfig.getMediaId()))
                .findFirst();
    }
}
