package com.bxm.adx.common.buy.dsp.domainmapping;

import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.DownloadInfo;
import com.google.common.net.InternetDomainName;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author fgf
 * @date 2025/6/6
 **/
@Configuration
public class AdxDomainMappingConfigCached implements AdxDomainMappingConfigChangeHandler {
    private static final String HTTP = "httt";
    private static final String HTTPS = "https";
    private final ConcurrentHashMap<String, Long> domainIdMappingMap = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, Long> bundleIdMappingMap = new ConcurrentHashMap<>();
    @Override
    public void doUpdate(AdxDomainMappingConfig old, AdxDomainMappingConfig latest) {
        if (isExecuteUpdateAndDoDelete(old, latest)) {
            if (Objects.isNull(latest.getMappingType()) || AdxDomainMappingConfig.DOMAIN == latest.getMappingType()) {
                // Update domain mapping
                domainIdMappingMap.put(latest.getDomain(), latest.getDomainId());
            } else if (AdxDomainMappingConfig.BUNDLE == latest.getMappingType()) {
                // Update bundle mapping
                bundleIdMappingMap.put(latest.getDomain(), latest.getDomainId());
            }
        }
    }

    @Override
    public void doDelete(AdxDomainMappingConfig old) {
        if (Objects.isNull(old.getMappingType()) || AdxDomainMappingConfig.DOMAIN == old.getMappingType()) {
            domainIdMappingMap.remove(old.getDomain());
        } else if (AdxDomainMappingConfig.BUNDLE == old.getMappingType()) {
            bundleIdMappingMap.remove(old.getDomain());
        }
    }

    public Long getDomainId(Bid bid) {
        String bundle = getBundle(bid);
        Long id = null;
        if (StringUtils.isNotBlank(bundle)) {
            id = bundleIdMappingMap.get(bundle);
        }
        if (Objects.nonNull(id)) {
            return id;
        }
        List<String> domainList = getDomain(bid);
        if (CollectionUtils.isEmpty(domainList)) {
            return id;
        }
        for (String domain : domainList) {
            if (StringUtils.isNotBlank(domain)) {
                id = domainIdMappingMap.get(domain);
                if (null == id) {
                    String topDomain = getTopDomain(domain);
                    if (StringUtils.isNotBlank(topDomain)) {
                        id = domainIdMappingMap.get(topDomain);
                    }
                }
            }
            if (Objects.nonNull(id)) {
                return id;
            }
        }
        return id;
    }

    /**
     * 获取域名
     * @param bid
     * @return 域名
     */
    public List<String> getDomain(Bid bid) {
        List<String> domainList = new ArrayList<>();
        String deeplink = bid.getDeep_link_url();
        if (StringUtils.isNotBlank(deeplink)) {
            String scheme = UriComponentsBuilder.fromUriString(deeplink).build().getScheme();
            if (HTTP.equals(scheme) || HTTPS.equals(scheme)) {
                scheme = UriComponentsBuilder.fromUriString(deeplink).build().getHost();
            }
            domainList.add(scheme);
        }
        String url = bid.getClick_through_url();
        if (StringUtils.isNotBlank(url)) {
            String host = UriComponentsBuilder.fromUriString(url).build().getHost();
            domainList.add(host);
        }
        String ulk = bid.getUniversal_link();
        if (StringUtils.isNotBlank(ulk)) {
            String host = UriComponentsBuilder.fromUriString(ulk).build().getHost();
            domainList.add(host);
        }
        String download = bid.getApp_download_url();
        if (StringUtils.isNotBlank(download)) {
            String host = UriComponentsBuilder.fromUriString(download).build().getHost();
            domainList.add(host);
        }
        String downloadUrl = Objects.nonNull(bid.getDownload_info()) ? bid.getDownload_info().getDownloadUrl() : null;
        if (StringUtils.isNotBlank(downloadUrl)) {
            String host = UriComponentsBuilder.fromUriString(downloadUrl).build().getHost();
            domainList.add(host);
        }
        return domainList;
    }

    /** * 获取顶级域名
     * @param host 域名
     * @return 顶级域名
     */
    public String getTopDomain(String host) {
        InternetDomainName domainName = InternetDomainName.from(host);
        if (domainName.hasPublicSuffix()) {
            InternetDomainName topPrivateDomain = domainName.topPrivateDomain();
            return topPrivateDomain.toString();
        }
        return host;
    }

    /** * 获取应用包名
     * @param bid
     * @return 应用包名
     */
    public String getBundle(Bid bid) {
        String bundle = bid.getBundle();
        DownloadInfo downloadInfo = bid.getDownload_info();
        if (Objects.nonNull(downloadInfo)) {
            if (StringUtils.isNotBlank(downloadInfo.getAppPackageName())) {
                bundle = downloadInfo.getAppPackageName();
            }
        }
        return bundle;
    }
}
