package com.bxm.adx.common.buy.position;

import com.alibaba.csp.sentinel.slots.block.flow.FlowRule;
import com.alibaba.csp.sentinel.slots.block.flow.FlowRuleManager;
import org.springframework.context.annotation.Configuration;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

/**
 * @author fgf
 * @date 2022/11/29
 **/
@Configuration
public class AdvertPointCached implements AdvertPointChangeHandler {
    /**
     * DSP广告端点集合
     * <pre>
     * |-------------|-----------------------------|
     * |    Key      | Value                       |
     * |-------------|-----------------------------|
     * | {dspId-dspPositionId}  |[adPoint,adPoint,...]|
     * |-------------|-----------------------------|
     * </pre>
     */
    private final ConcurrentHashMap<String, AdvertPoint> advertPointMap = new ConcurrentHashMap<>();

    @Override
    public void doUpdate(AdvertPoint old, AdvertPoint latest) {
        if (isExecuteUpdateAndDoDelete(old, latest)) {
            advertPointMap.put(latest.getUniqueKey(), latest);
            initFlowRules(latest);
        }
    }

    @Override
    public void doDelete(AdvertPoint old) {
        advertPointMap.remove(old.getUniqueKey());
    }


    public AdvertPoint getAdPoint(String dspId, String appId, String dspPositionId) {
        return advertPointMap.get(getMapKey(dspId, appId, dspPositionId));
    }


    private static String getMapKey(String dspId, String appId, String dspPositionId) {
        return dspId + "-" + appId + "-" + dspPositionId;
    }

    /**
     * 更新端点qps限制规则
     *
     * @param advertPoint
     */
    private void initFlowRules(AdvertPoint advertPoint) {
        if (Objects.isNull(advertPoint.getQpsLimit())) {
            return;
        }
        List<FlowRule> current = FlowRuleManager.getRules();
        List<FlowRule> rules = new ArrayList<>(current);
        FlowRule rule = new FlowRule();
        rule.setResource(advertPoint.getUniqueKey());
        rule.setGrade(com.alibaba.csp.sentinel.slots.block.RuleConstant.FLOW_GRADE_QPS);
        rule.setCount(advertPoint.getQpsLimit());
        rules.add(rule);
        FlowRuleManager.loadRules(rules);
    }
}
