package com.bxm.adx.common.market.exchange.rebuild.filter;

import com.bxm.adx.common.CacheKeys;
import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.limiter.DeleteExpiredHandler;
import com.bxm.adx.common.market.exchange.ExchangeContext;
import com.bxm.adx.common.market.exchange.rebuild.response.ResponseBuildAttribute;
import com.bxm.adx.common.rule.Rule;
import com.bxm.adx.common.rule.WhiteAndBlackRule;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Configuration;

import java.util.*;

/**
 * 人群包定向
 * @author fgf
 * @date 2025/2/17
 **/
@Slf4j
@Configuration
public class CrowdPackageDispatcherPriceConfigFilter implements DispatcherPriceConfigFilter {

    private final Fetcher fetcher;
    private final DeleteExpiredHandler deleteExpiredHandler;

    public CrowdPackageDispatcherPriceConfigFilter(@Qualifier("crowdPackFetcher") Fetcher fetcher, DeleteExpiredHandler deleteExpiredHandler) {
        this.fetcher = fetcher;
        this.deleteExpiredHandler = deleteExpiredHandler;
    }

    @Override
    public void filter(Collection<DispatcherPriceConfig> priceConfigs, Bid bid, SeatBid seatBid, ResponseBuildAttribute attribute) {
        Set<String> cache = ExchangeContext.getCrowdPackageSet();
        Set<String> crowdPackageIds = Objects.nonNull(cache) ? cache : getUserCrowdPackageSet(attribute.getBidRequest());
        priceConfigs.removeIf(config -> !hit(config, crowdPackageIds));
    }

    @Override
    public int getOrder() {
        return 50;
    }

    private boolean hit(DispatcherPriceConfig config, Set<String> crowdPackageIds) {
        Rule rule = config.getCrowdPackageOrientation();
        if (null == rule) {
            return true;
        }
        WhiteAndBlackRule whiteAndBlackRule = new WhiteAndBlackRule(rule);
        WhiteAndBlackRule.WhiteAndBlack whiteAndBlackSet = whiteAndBlackRule.getWhiteAndBlack();
        Set<String> whiteSet = whiteAndBlackSet.getWhite();
        if (CollectionUtils.isNotEmpty(whiteSet)) {
            if (CollectionUtils.isEmpty(crowdPackageIds)) {
                return false;
            }
            for (String crowdPackageId : crowdPackageIds) {
                if (whiteSet.contains(crowdPackageId)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 获取用户命中的人群包id集合
     *
     * @param request
     * @return
     */
    protected Set<String> getUserCrowdPackageSet(BidRequest request) {
        Set<String> userCrowdPackageSet = new HashSet<>();
        KeyGenerator key = CacheKeys.getCrowdPackageNewKey(request.getDevice());
        if (Objects.nonNull(key)) {
            if (log.isDebugEnabled()) {
                log.debug("key {}", key.generateKey());
            }
            long currentSecond = System.currentTimeMillis() / 1000;
            Map<String, Long> idAndExpireSecondMap = fetcher.hfetchall(key, Long.class);
            if (MapUtils.isNotEmpty(idAndExpireSecondMap)) {
                for (Map.Entry<String, Long> entry : idAndExpireSecondMap.entrySet()) {
                    //key为人群包id，value为过期时间（时间戳秒）
                    String id = entry.getKey();
                    if (entry.getValue() > currentSecond) {
                        userCrowdPackageSet.add(id);
                    } else {
                        deleteExpiredHandler.delete(key, id);
                    }
                }
            }
        }
        return userCrowdPackageSet;
    }
}
