package com.bxm.adx.common.market.exchange.rebuild.filter;

import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.market.exchange.rebuild.response.ResponseBuildAttribute;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.context.annotation.Configuration;

import java.util.Collection;
import java.util.List;

/**
 * 任务ID/质量分出价过滤
 * @author jingyu.li
 * @date 2024-06-06
 */
@Slf4j
@Configuration
public class TaskIdDispatcherPriceConfigFilter implements DispatcherPriceConfigFilter{

    @Override
    public int getOrder() {
        return 10;
    }

    @Override
    public void filter(Collection<DispatcherPriceConfig> priceConfigs, Bid bid, SeatBid seatBid, ResponseBuildAttribute attribute) {
        //根据taskId、质量分匹配白名单 如果白名单过滤策略开启 并且 不在白名单内 的出价就删除
        String taskId = bid.getTask_id();
        //用户质量分
        Integer userScoreLevel = bid.getUser_score_level();
        if (log.isDebugEnabled()) {
            log.debug(" adid:{},taskId:{},userScoreLevel:{}", bid.getAdid(), taskId,userScoreLevel);
        }
        //过滤
        priceConfigs.removeIf(config -> config.getFilterMode() != null && DispatcherPriceConfig.FILTER_MODEL_WHITE == config.getFilterMode() && !getPriceConfig(config, taskId, userScoreLevel));
    }

    /**
     * 是否命中当前出价
     *
     * @param config 出价配置
     * @param taskId 任务id
     * @param userScoreLevel 用户质量分
     * @return true、false
     */
    private boolean getPriceConfig(DispatcherPriceConfig config, String taskId, Integer userScoreLevel) {
        List<String> taskIds = config.getTaskIds();
        if (CollectionUtils.isEmpty(taskIds)) {
            return false;
        }
        String interval = locateInterval(userScoreLevel);
        return taskIds.contains(taskId) || taskIds.contains(interval) || taskIds.contains(taskId + ":" + interval);
    }

    /**
     * 根据分数转换区间
     * @param number 分数
     * @return 区间
     */
    public String locateInterval(Integer number) {
        if (number == null) {
            return "";
        }

        if (number >= 81 && number <= 100) {
            return "81_100";
        } else if (number >= 61 && number <= 80) {
            return "61_80";
        } else if (number >= 41 && number <= 60) {
            return "41_60";
        } else if (number >= 21 && number <= 40) {
            return "21_40";
        } else if (number >= 1 && number <= 20) {
            return "1_20";
        } else {
            return String.valueOf(number);
        }
    }
}
