package com.bxm.adx.common.market.exchange.rebuild.offer;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.utils.MurmursUtils;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.xcache.Fetcher;
import com.bxm.warcar.xcache.TargetFactory;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 不设置(动态出价)
 *
 * @author fgf
 * @date 2023/1/10
 **/
@Slf4j
@Configuration
public class NoneDspAvgPriceOffer implements Offer {

    private static final int MURMURS_SEED = 0x1234ABCD;
    private final Fetcher fetcher;

    public NoneDspAvgPriceOffer(Fetcher fetcher) {
        this.fetcher = fetcher;
    }

    @Override
    public OfferResult offer(Bid bid, BidRequest request, Dispatcher dispatcher, DispatcherPriceConfig priceConfig) {
        return OfferResult.builder()
                .offer(getPriceByConfig(bid, request, priceConfig))
                .chargeType(AdxConstants.ChargeType.CPM)
                .budgetType(AdxConstants.ChargeType.CPM)
                .build();
    }

    @Override
    public OfferType offerType() {
        return OfferType.NONE_Dsp_Avg_Price;
    }

    /**
     * 获取配置出价配置的情况下的出价
     *
     * @param bidRequest
     * @param bid
     * @param config
     * @return
     */
    private BigDecimal getPriceByConfig(Bid bid, BidRequest bidRequest, DispatcherPriceConfig config) {
        if (Objects.isNull(config)) {
            return bid.getPrice();
        }
        //媒体出价=dsp出价*（1-利润率）*出价系数 注:此处dsp出价是经过流量分配中广告主系数处理后的出价，非原始dsp出价
        BigDecimal dspWinPrice = Optional.ofNullable(bid.getDspWinPrice())
                .orElse(Optional.ofNullable(bid.getDsp_dis_price()).orElse(bid.getDsp_price()));
//        BigDecimal profitMargin = Optional.ofNullable(config.getProfitMargin()).orElse(BigDecimal.ZERO);
        ProfitMarginGroup profitMarginGroup = getProfitMargin(bidRequest, config);
        if (Objects.nonNull(profitMarginGroup.groupId)) {
            bid.setDpc_strategy_id(getExpId(config) + "-" + profitMarginGroup.groupId);
        }
        BigDecimal profitMargin = profitMarginGroup.getProfitMargin();
        BigDecimal newPrice = dspWinPrice
                .multiply(new BigDecimal(100.00).subtract(profitMargin))
                .multiply(config.getBiddingCoefficient()).movePointLeft(2);
//        BigDecimal maxBidFloor = maxBidFloor(bidRequest, dispatcher);
        BigDecimal bidFloor = bidRequest.getImps().iterator().next().getBid_floor();
        if (newPrice.compareTo(bidFloor) >= 0) {
            return newPrice.setScale(0, RoundingMode.HALF_UP);
        } else {
            //探索出价-底价加1分钱
            BigDecimal explorePrice = bidFloor.add(BigDecimal.ONE);
            if (dspWinPrice.compareTo(explorePrice) >= 0) {
                return explorePrice.setScale(0, RoundingMode.HALF_UP);
            }
        }

        return null;
    }

    /**
     * 获取分组利润率
     *
     * @param config
     * @return
     */
    private ProfitMarginGroup getProfitMargin(BidRequest request, DispatcherPriceConfig config) {
        Map<Integer, BigDecimal> profitMarginMap = config.getProfitMarginGroupMap();
        if (Objects.isNull(profitMarginMap) || profitMarginMap.isEmpty()) {
            return new ProfitMarginGroup(null, Optional.ofNullable(config.getProfitMargin()).orElse(BigDecimal.ZERO));
        }

        Map<String, Integer> groupMap = fetcher.hfetchall(new TargetFactory<Integer>()
                .keyGenerator(strategyInfoKey(config))
                .cls(Integer.class)
                .skipNativeCache(false)
                .build());

        int per = (int) Math.abs(MurmursUtils.hash(request.getId(), MURMURS_SEED) % 100);
        if (Objects.isNull(groupMap) || groupMap.isEmpty()) {
            int groupId = (per / 20);
            BigDecimal pm = profitMarginMap.get(groupId);
            if (Objects.isNull(pm)) {
                log.error("profit margin is null");
                throw new AdxException("profit margin is null");
            }
            return new ProfitMarginGroup(groupId, pm);
        } else {
            int scale = 0;
            for (int i = 0; i < profitMarginMap.size(); i++) {
                scale = scale + groupMap.get(i + "");
                if (per < scale) {
                    return new ProfitMarginGroup(i, profitMarginMap.get(i));
                }
            }
            log.error("profit margin is null");
            throw new AdxException("profit margin is null");
        }
    }

    static class ProfitMarginGroup {
        private Integer groupId;
        private BigDecimal profitMargin;

        public ProfitMarginGroup(Integer groupId, BigDecimal profitMargin) {
            this.groupId = groupId;
            this.profitMargin = profitMargin;
        }

        public Integer getGroupId() {
            return groupId;
        }

        public void setGroupId(Integer groupId) {
            this.groupId = groupId;
        }

        public BigDecimal getProfitMargin() {
            return profitMargin;
        }

        public void setProfitMargin(BigDecimal profitMargin) {
            this.profitMargin = profitMargin;
        }
    }

    /**
     * 存放数据计算的分组比例结果
     * @param config
     * @return
     */
    private static KeyGenerator strategyInfoKey(DispatcherPriceConfig config) {
        String expId = getExpId(config);
        return () -> KeyBuilder.build("ADX", "DPC", "STRATEGY", config.getPriceModeId(), expId);
    }

    private static String getExpId(DispatcherPriceConfig config) {
        BigDecimal lower = config.getProfitMarginLower().setScale(2, RoundingMode.HALF_UP);
        BigDecimal upper = config.getProfitMarginUpper().setScale(2, RoundingMode.HALF_UP);
        String expId = lower.toString() + "_" + upper.toString();
        return expId;
    }

    /**
     * since 20230522 逻辑修改，不取两者最大值，直接和原始媒体底价做比较
     * <p>
     * 取配置底价和请求底价中较大的那个
     *
     * @param bidRequest
     * @param dispatcher
     * @return
     */
    @Deprecated
    private BigDecimal maxBidFloor(BidRequest bidRequest, Dispatcher dispatcher) {
        //元转分
        BigDecimal dispatcherBidFloor = Optional.ofNullable(dispatcher.getDspBasePrice()).orElse(BigDecimal.ZERO).movePointRight(2);
        BigDecimal bidFloor = bidRequest.getImps().iterator().next().getBid_floor();
        int result = bidFloor.compareTo(dispatcherBidFloor);
        switch (result) {
            case -1:
                return dispatcherBidFloor;
            case 0:
            case 1:
            default:
                return bidFloor;
        }
    }
}
