package com.bxm.adx.common.market.exchange.rebuild.response;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.OpenlogConstants;
import com.bxm.adx.common.adapter.AbstractPluginBuyModelAdapter;
import com.bxm.adx.common.adapter.BuyModelAdapter;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.ClickMonitor;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.warcar.integration.pair.Pair;
import com.bxm.warcar.utils.UrlHelper;
import com.bxm.warcar.utils.device.SimpleDevice;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * 回传转化链接处理
 * 在点击中替换回传链接宏参
 *
 * @author fgf
 * @date 2024/9/12
 **/
@Slf4j
@Configuration
public class ConversionBuilder implements AdxBidResponseBuilder {
    private static final int PANGU_DSP_ID = 105;
    private static final int QIHANG_DSP_ID = 97;
    private final AdxProperties adxProperties;
    private final Pair pair;
    private final static String KEY = "adx.ocpx.conversion.map";
    public ConversionBuilder(AdxProperties adxProperties, Pair pair) {
        this.adxProperties = adxProperties;
        this.pair = pair;
    }

    @Override
    public void rebuildAdxBidResponse(BidResponse bidResponse, ResponseBuildAttribute attribute) {
        Buyer buyer = attribute.getBuyer();
        BuyModelAdapter adapter = buyer.getModelAdapter();
        if (!(adapter instanceof AbstractPluginBuyModelAdapter)) {
            return;
        }
        AbstractPluginBuyModelAdapter pluginAdapter = (AbstractPluginBuyModelAdapter) adapter;
        String macro = pluginAdapter.getPluginConfig().getProperties().getProperty(AdxConstants.PluginParam.CALLBACK_URL);
        if (StringUtils.isBlank(macro)) {
            return;
        }
        for (SeatBid seatBid : bidResponse.getSeat_bid()) {
            for (Bid bid : seatBid.getBid()) {
                List<ClickMonitor> clickMonitors = bid.getClick_monitors();
                if (CollectionUtils.isEmpty(clickMonitors)) {
                    continue;
                }
                clickMonitors.forEach(
                        clickMonitor -> {
                            MultiValueMap<String, String> params = buildCustomizeParams(bidResponse, seatBid, bid, attribute);
                            String url = buildConversionUrl(params);
                            String click = clickMonitor.getClick_monitor_url().replace(macro, UrlHelper.urlEncode(url));
                            clickMonitor.setClick_monitor_url(click);
                        }
                );
            }
        }
    }

    @Override
    public int getOrder() {
        return Integer.MAX_VALUE;
    }

    /**
     * 构建回传监测链接
     *
     * @param params
     * @return
     */
    private String buildConversionUrl(MultiValueMap<String, String> params) {
        String url = adxProperties.getOpenLog().createBase(OpenlogConstants.Mt.CONVERSION, params);
        return url;
    }

    /**
     * 构建客制化埋点参数
     *
     * @param bidResponse
     * @return
     */
    private MultiValueMap<String, String> buildCustomizeParams(BidResponse bidResponse,
                                                               SeatBid seatBid,
                                                               Bid bid,
                                                               ResponseBuildAttribute attribute) {
        MultiValueMap<String, String> customizeParams = new LinkedMultiValueMap<>();
        String media = attribute.getBidRequest().getMediaId();
        String bundle = bid.getBundle();
        String bidid = attribute.getBidRequest().getId();
        String rt = System.currentTimeMillis() / 1000 + "";
        //此处和bes返回中的crid是否有关系
        String device = getDevice(attribute);

        customizeParams.add(Macros.Media, media);
        customizeParams.add(Macros.Bidid, bidid);
        customizeParams.add(Macros.Bundle, bundle);
        customizeParams.add(Macros.RequestTime, rt);
        customizeParams.add(Macros.Device, device);

        Long dspId = attribute.getBuyer().getDsp().getId();
        customizeParams.add(Macros.Advertiser, dspId.toString());
        customizeParams.add(Macros.ConversionDef, getConDef(Long.valueOf(media), dspId));
        if (Objects.nonNull(bid.getDpc_strategy_id())){
            customizeParams.add(OpenlogConstants.MacrosCustomizeParams.DPC_STRATEGY_ID, UrlHelper.urlEncode(bid.getDpc_strategy_id()));
        }
        return customizeParams;
    }

    private String getDevice(ResponseBuildAttribute attribute) {
        Device device = attribute.getBidRequest().getDevice();
        SimpleDevice simpleDevice = SimpleDevice.builder()
                .imei(device.getImei())
                .imeiMd5(device.getImei_md5())
                .androidid(device.getDpid())
                .androididMd5(device.getDpid_md5())
                .oaid(device.getOaid())
                .oaidMd5(device.getOaid_md5())
                .idfa(device.getIdfa())
                .idfaMd5(device.getIdfa_md5())
                .gaid(device.getGaid())
                .gaidMd5(device.getGaid_md5())
                .ipv4(device.getIp())
                .ipv6(device.getIp_v6())
                .caid(device.getCaidInfoStr())
                .build();
        return SimpleDevice.Base64Cipher.encrypt(simpleDevice);
    }

    /**
     * 获取广告主回传类型和媒体映射关系
     * @param mediaId
     * @param dspId
     * @return
     */
    private String getConDef(Long mediaId, Long dspId) {
        String config = pair.get(KEY).of();
        if (StringUtils.isNotBlank(config)) {
            List<ConversionData> conversionDataList = JSON.parseObject(config, new TypeReference<List<ConversionData>>() {});
            Optional<ConversionData> optional = conversionDataList.stream()
                    .filter(conversionData -> dspId.equals(conversionData.getDspId()) && mediaId.equals(conversionData.getMediaId()))
                    .findFirst();
            if (optional.isPresent()) {
                return optional.get().getConvDef();
            }
        }
        return StringUtils.EMPTY;
    }

    /**
     * @link https://bxmrds.yuque.com/mizhsy/eqfqab/grgl67#t2Jdr
     * FeedBackEntity数据结构
     */
    interface Macros {
        String Advertiser = "dspid";
        String Media = "medid";
        String Bundle = "advb";
        String Bidid = "bidid";
        String RequestTime = "rt";
        String Device = "device";
        String ConversionDef = "conv_def";
    }

    @Data
    static class ConversionData {
        private Long mediaId;
        private Long dspId;
        private String convDef;
    }
}
