package com.bxm.adx.common.market.exchange.rebuild.response;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.market.exchange.rebuild.offer.OfferFactory;
import com.bxm.adx.common.market.exchange.rebuild.offer.OfferResult;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.warcar.integration.pair.Pair;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Map;
import java.util.Set;

/**
 * 媒体出价
 *
 * @author fgf
 * @date 2023/1/10
 **/
@Slf4j
@Configuration
public class OfferPriceBuilder implements AdxBidResponseBuilder {
    private final static Long HUAWEI_DSP_ID = 128L;
    private final static String OFFER_CONFIG_KEY = "offer.config";
    private final Pair pair;
    private final OfferFactory factory;

    public OfferPriceBuilder(Pair pair, OfferFactory factory) {
        this.pair = pair;
        this.factory = factory;
    }

    @Override
    public int getOrder() {
        return 5;
    }

    @Override
    public void rebuildAdxBidResponse(BidResponse bidResponse, ResponseBuildAttribute attribute) {
        OfferResult offerConfigResult = getOfferResult(attribute);
        if (null != offerConfigResult) {
            for (SeatBid seatBid : bidResponse.getSeat_bid()) {
                for (Bid bid : seatBid.getBid()) {
                    bid.setCharge_type(offerConfigResult.getChargeType());
                    bid.setBudget_type(offerConfigResult.getBudgetType());
                    bid.setPrice(offerConfigResult.getOffer());
                }
            }
        } else {
            for (SeatBid seatBid : bidResponse.getSeat_bid()) {
                Map<Bid, OfferResult> offerResultMap = factory.offer(seatBid.getBid(), attribute);
                for (Bid bid : seatBid.getBid()) {
                    OfferResult result = offerResultMap.get(bid);
                    if (result == null) {
                        continue;
                    }
                    int chargeType = result.getChargeType();
                    bid.setCharge_type(chargeType);
                    int budgetType = result.getBudgetType();
                    bid.setBudget_type(budgetType);

                    switch (budgetType) {
                        case AdxConstants
                                .ChargeType.CPM:
                            bid.setPrice(result.getOffer());
                            bid.setBid(null);
                            break;
                        case AdxConstants
                                .ChargeType.CPC:
                            bid.setBid(result.getOffer());
                            bid.setPrice(null);
                            break;
                        case AdxConstants.ChargeType.OCPM:
                            bid.setObid(result.getOffer());
                            bid.setPrice(null);
                            bid.setBid(null);
                            break;
                        default:
                            log.warn("unsupported budgetType, dispatcher {}", attribute.getDispatcher().getId());
                            throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
                    }
                }
            }
        }
    }

    /**
     * 华为DSP贴底价出价
     * @param attribute
     * @return
     */
    private OfferResult getOfferResult(ResponseBuildAttribute attribute) {
        Dispatcher dispatcher = attribute.getDispatcher();
        Long dspId = dispatcher.getDspId();
        String dspTagId = dispatcher.getDspPosid();
        if (HUAWEI_DSP_ID.equals(dspId)) {
            Set<String> configs = pair.get(OFFER_CONFIG_KEY).ofHashSet();
            if (null == configs || configs.isEmpty()) {
                return null;
            }
            for (String config : configs) {
                String[] params = config.split("-");
                if (params.length != 4) {
                    continue;
                }
                String tagId = params[0];
                String offer = params[1];
                String limit = params[2];
                String lowest = params[3];
                if (StringUtils.isBlank(tagId) || StringUtils.isBlank(offer) || StringUtils.isBlank(limit) || StringUtils.isBlank(lowest)) {
                    continue;
                }
                if (tagId.equals(dspTagId)) {
                    BigDecimal floor = attribute.getBidRequest().getImps().iterator().next().getBid_floor();
                    if (null == floor) {
                        floor = BigDecimal.ZERO;
                    }
                    BigDecimal offerPrice = new BigDecimal(offer).add(floor);
                    BigDecimal upperLimit = new BigDecimal(limit);
                    BigDecimal lowerLimit = new BigDecimal(lowest);
                    if (offerPrice.compareTo(upperLimit) > 0) {
                        offerPrice = upperLimit;
                    } else {
                        if (offerPrice.compareTo(lowerLimit) < 0) {
                            offerPrice = lowerLimit;
                        }
                    }
                    return OfferResult.builder()
                            .budgetType(AdxConstants.ChargeType.CPM)
                            .chargeType(AdxConstants.ChargeType.CPM)
                            .offer(offerPrice.setScale(0, RoundingMode.HALF_UP))
                            .build();
                }
            }
        }
        return null;
    }
}
