package com.bxm.adx.common.market.filter;

import com.alibaba.fastjson.JSONObject;
import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.warcar.integration.pair.Pair;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

/**
 * 快手低价过滤器
 * 只支持单个seatbid
 *
 * @author allen
 * @since 2019-12-18
 */
@Slf4j
@Configuration
public class KuaishouPriceLowerFilter implements Filter {
    private final static List<String> KUAI_SHOU_MEDIA_ID = Lists.newArrayList("87", "100");
    private final Pair pair;
    private final static String POSITION_CTR_KEY = "position.ctr.config";

    public KuaishouPriceLowerFilter(Pair pair) {
        this.pair = pair;
    }

    @Override
    public void filter(List<Deal> deals, Set<Deal> trash) {
        BidRequest request = AdxContextFactory.get().getBidRequest();
        String mediaId = request.getMediaId();
        if (KUAI_SHOU_MEDIA_ID.contains(mediaId)) {
            handleKuaishouDeals(deals,trash);
        }
    }

    @Override
    public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE;
    }

    /**
     * 快手媒体特殊逻辑处理
     */
    private void handleKuaishouDeals(List<Deal> deals, Set<Deal> trash) {
        String tagId = AdxContextFactory.get().getBidRequest().getBxmTagId();

        Optional<Bid> topBid = deals.stream()
                .filter(deal -> Objects.nonNull(deal.getBidResponse()))
                .map(Deal::getBidResponse)
                .filter(bidResponse -> CollectionUtils.isNotEmpty(bidResponse.getSeat_bid()))
                .map(BidResponse::getSeat_bid)
                .flatMap(Collection::stream)
                .filter(seatBid -> CollectionUtils.isNotEmpty(seatBid.getBid()))
                .map(SeatBid::getBid)
                .flatMap(Collection::stream)
                .max(Comparator.comparing(bid -> calculateEffectiveCpm(bid, tagId)));
        if (!topBid.isPresent()) {
            deals.removeIf(deal -> {
                return true;
            });
            return;
        }
        Bid winningBid = topBid.get();

        filterDeals(deals, trash, winningBid);
    }

    /**
     * 计算 CPM（适配 CPC 和 CPM 出价）
     * POSITION_CTR_KEY value 格式：{"802779054005":"0.02"}
     */
    private BigDecimal calculateEffectiveCpm(Bid bid, String positionId) {
        if (AdxConstants.ChargeType.CPM == bid.getCharge_type()) {
            return bid.getPrice();
        } else if (AdxConstants.ChargeType.CPC == bid.getCharge_type()) {
            String positionCtr = pair.get(POSITION_CTR_KEY).of();
            if (StringUtils.isNotBlank(positionCtr)){
                Map<String, String> map = JSONObject.parseObject(positionCtr, Map.class);
                String value = map.getOrDefault(positionId, "0.01");
                if (StringUtils.isNotBlank(value)){
                    BigDecimal ctr = new BigDecimal(value);
                    return bid.getBid().multiply(ctr).multiply(BigDecimal.valueOf(1000));
                }
            }
            return bid.getBid().multiply(new BigDecimal("0.01")).multiply(BigDecimal.valueOf(1000));
        }
        return BigDecimal.ZERO;
    }

    private void filterDeals(List<Deal> deals, Set<Deal> trash, Bid topBid) {
        deals.removeIf(deal -> {
            BidResponse response = deal.getBidResponse();
            List<SeatBid> seatBids = response.getSeat_bid();

            seatBids.removeIf(seatBid -> {
                List<Bid> bidList = seatBid.getBid();
                bidList.removeIf(bid -> !bid.equals(topBid));
                return CollectionUtils.isEmpty(bidList);
            });

            boolean isDealEmpty = CollectionUtils.isEmpty(seatBids);
            if (isDealEmpty) {
                trash.add(deal);
            }
            return isDealEmpty;
        });
    }
}
