package com.bxm.adx.common.market.monitor;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.report.*;
import com.bxm.adx.common.sell.builder.BuildAttribute;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.ClickMonitor;
import com.bxm.adx.common.sell.response.ImpMonitor;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @author fgf
 * @date 2023/4/14
 **/
@Slf4j
public class ServerReportHandler implements MonitorHandler {
    private final ServerReportService serverReportService;
    private final AttributionReportService attributionReportService;
    private final Pattern pattern = Pattern.compile("(?<=\\.)[^.]+\\.[^.]+$");
    public ServerReportHandler(ServerReportService serverReportService, AttributionReportService attributionReportService) {
        this.serverReportService = serverReportService;
        this.attributionReportService = attributionReportService;
    }

    @Override
    public int getOrder() {
        return 0;
    }

    @Override
    public void handler(BuildAttribute attribute) {
        Bid bid = attribute.getBid();
        //是否需要服务端上报
        ServerReportConfig config = serverReportService.getConfig(attribute.getMediaId(), attribute.getAppId(),
                attribute.getDspId().toString(), attribute.getSspRequest().getId());
        if (Objects.isNull(config)) {
            return;
        }
        int limit = config.getLimit();

        List<ImpMonitor> impMonitors = bid.getImp_monitors();
        List<String> impUrls = getImpUrls(impMonitors);
        if (isExceed(impUrls, limit)) {
            //获取超出限制的条数并存储,等之后服务端上报
            List<String> exceedImp = getExceed(impUrls, config, ReportType.IMP, attribute);
            //移除
            impMonitors.removeIf(
                    impMonitor -> {
                        return exceedImp.contains(impMonitor.getImp_monitor_url());
                    }
            );
        }

        List<ClickMonitor> clickMonitors = bid.getClick_monitors();
        List<String> clickUrls = getClickUrls(clickMonitors);
        if (isExceed(clickUrls, limit)) {
            List<String> exceedClick = getExceed(clickUrls, config, ReportType.CLICK, attribute);
            //归因
            attribution(exceedClick, attribute);
            clickMonitors.removeIf(
                    clickMonitor -> {
                        return exceedClick.contains(clickMonitor.getClick_monitor_url());
                    }
            );
        }
    }

    /**
     * 点击归因处理
     * 只支持CPM计费的媒体
     * 限制支持点击1条
     * @param exceedClick
     * @param attribute
     */
    private void attribution(List<String> exceedClick, BuildAttribute attribute) {
        try {
            Long dspId = attribute.getDspId();
            Integer mediaId = Integer.valueOf(attribute.getMediaId());
            Integer chargeType = attribute.getMediaChargeType();
            if (AdxConstants.ChargeType.CPM != chargeType) {
                return;
            }
            if (AttributionReportService.SUPPORT_DSP.contains(dspId)
                    && AttributionReportService.SUPPORT_MEDIA.contains(mediaId)) {
                int size = exceedClick.size();
                if (size != 1) {
                    log.warn("unsupport size = {}, tagId = {}, dTagId = {}", size, attribute.getTagId(), attribute.getdTagId());
                    return;
                }
                String url = exceedClick.get(0);
                attributionReportService.saveClickUrl(attribute.getSspRequest().getId(), url);
            }
        } catch (Exception e) {
            log.error("ServerReportHandler attribution error", e);
        }
    }

    /**
     * 获取曝光链接
     *
     * @param impMonitors
     * @return
     */
    private List<String> getImpUrls(List<ImpMonitor> impMonitors) {
        if (CollectionUtils.isEmpty(impMonitors)) {
            return Lists.newArrayList();
        }
        List<String> urls = impMonitors.stream()
                .filter(impMonitor -> StringUtils.isNotEmpty(impMonitor.getImp_monitor_url()))
                .map(ImpMonitor::getImp_monitor_url)
                .collect(Collectors.toList());
        return urls;
    }

    /**
     * 获取点击链接
     *
     * @param clickMonitors
     * @return
     */
    private List<String> getClickUrls(List<ClickMonitor> clickMonitors) {
        if (CollectionUtils.isEmpty(clickMonitors)) {
            return Lists.newArrayList();
        }
        List<String> urls = clickMonitors.stream()
                .filter(clickMonitor -> StringUtils.isNotEmpty(clickMonitor.getClick_monitor_url()))
                .map(ClickMonitor::getClick_monitor_url)
                .collect(Collectors.toList());
        return urls;
    }

    private boolean isExceed(List<String> urls, int limit) {
        return (urls.size() - limit) > 0;
    }

    /**
     * 获取超出限制条数的监测地址
     *
     * @param urls
     * @param config
     * @param type
     * @return
     */
    private List<String> getExceed(List<String> urls, ServerReportConfig config, ReportType type, BuildAttribute attribute) {
        List<String> domains = config.getDomains();
        if (CollectionUtils.isNotEmpty(domains)) {
            urls.sort(comparator(domains));
        }
        List<String> exceed = urls.subList(config.getLimit(), urls.size());
        ReportParam reportParam = buildReportParam(config, type, attribute);
        serverReportService.saveExceed(reportParam, exceed);
        return exceed;
    }

    /**
     * 上报参数
     * @param attribute
     * @return
     */
    private ReportParam buildReportParam(ServerReportConfig config, ReportType type, BuildAttribute attribute) {
        Bid bid = attribute.getBid();
        String adId = bid.getAdid();
        if (StringUtils.isEmpty(adId)) {
            log.warn("adid is empty");
        }
        Integer chargeType = bid.getCharge_type();
        if (Objects.isNull(chargeType)) {
            log.warn("charge-type is empty");
        }
        return ReportParam.builder()
                .type(type)
                .bidId(config.getBidId())
                .appId(config.getAppId())
                .dspId(config.getDspId())
                .mediaId(config.getMediaId())
                .adId(adId)
                .bidModel(attribute.getSspRequest().getBid_model())
                .chargeType(chargeType.toString())
                .build();
    }

    /**
     * 通过域名识别广告主, 把广告主的监测优先媒体上报
     *
     * @param domains
     * @return
     */
    private Comparator<String> comparator(List<String> domains) {
        return new Comparator<String>() {
            @Override
            public int compare(String o1, String o2) {
                String domain1 = getDomain(UriComponentsBuilder.fromUriString(o1).build().getHost());
                String domain2 = getDomain(UriComponentsBuilder.fromUriString(o2).build().getHost());
                boolean b1 = domains.contains(domain1);
                boolean b2 = domains.contains(domain2);
                if (b1) {
                    if (b2) {
                        return 0;
                    } else {
                        return -1;
                    }
                } else {
                    if (b2) {
                        return 1;
                    } else {
                        return 0;
                    }
                }
            }
        };
    }

    private String getDomain(String host) {
        Matcher matcher = pattern.matcher(host);
        if (matcher.find()) {
            String domain = matcher.group();
            return domain;
        }
        return "";
    }
}
