package com.bxm.adx.common.market.nbr;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.CacheKeys;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.limiter.InsertExpireHandler;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.warcar.cache.KeyGenerator;
import com.google.common.collect.Lists;
import lombok.Data;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.RandomUtils;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * 用于获取dsp的具体tagId的参竞人群包
 * 注：屏蔽此tagId的参竞，不向媒体返回（此步骤在插件中处理）
 *
 * @author fgf
 * @date 2024/8/9
 **/
public class BidDeviceHandler implements NoBidResponseHandler {
    private final InsertExpireHandler insertExpireHandler;

    public BidDeviceHandler(InsertExpireHandler insertExpireHandler) {
        this.insertExpireHandler = insertExpireHandler;
    }

    @Override
    public void handler(Deal... deals) {
        if (ArrayUtils.isEmpty(deals)) {
            return;
        }
        KeyGenerator key = keyGenerator(deals);
        if (Objects.isNull(key)) {
            return;
        }

        for (Deal deal : deals) {
            String crowdPackId = crowdPackId(deal);
            if (StringUtils.isBlank(crowdPackId)) {
                continue;
            }
            Long expireVal = deal.getBidResponse().getCdt();
            if (Objects.isNull(expireVal)) {
                continue;
            }
            long keyExpire = expireVal.longValue() - (System.currentTimeMillis() / 1000);
            //使用最新的过期时间
            if (keyExpire > 0) {
                //防止集中过期导致redis负载升高
                keyExpire = keyExpire + RandomUtils.nextInt(1, 600);
                insertExpireHandler.insert(key, crowdPackId, String.valueOf(expireVal), (int) keyExpire);
            }
        }
    }

    @Override
    public int nbr() {
        return AdxConstants.Nbr.BID_DEVICE;
    }

    /**
     * 获取对应的人群包id
     * @param deal
     * @return
     */
    private String crowdPackId(Deal deal) {
        List<Config> configs = Lists.newArrayList(new Config());
        Dispatcher dispatcher = deal.getDispatcher();
        Long dspId = dispatcher.getDspId();
        String tagId = dispatcher.getDspPosid();
        Optional<Config> optional = configs.stream().filter(
                config -> {
                    Long configDspId = config.getDspId();
                    String configTagId = config.getTagId();
                    if (dspId.equals(configDspId) && tagId.equals(configTagId)) {
                        return true;
                    }
                    return false;
                }
        ).findFirst();
        if (optional.isPresent()) {
            return optional.get().getCrowId();
        }
        return StringUtils.EMPTY;
    }

    /**
     * 获取redis-key
     *
     * @param deals
     * @return
     */
    private KeyGenerator keyGenerator(Deal... deals) {
        for (Deal deal : deals) {
            BidRequest request = deal.getBidRequest();
            if (Objects.nonNull(request)) {
                return CacheKeys.getCrowdPackageNewKey(request.getDevice());
            }
        }
        return null;
    }

    @Data
    static class Config {
        private Long dspId = 97L;
        private String tagId = "bxm_djk_7";
        private String crowId = "185";
    }
}
