package com.bxm.adx.common.openlog.listener.internal;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.DotEventControlProperties;
import com.bxm.adx.common.OpenlogConstants;
import com.bxm.adx.common.buy.dsp.domainmapping.AdxDomainMappingConfigCached;
import com.bxm.adx.common.openlog.event.internal.DspBidEvent;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.builder.BuildAttribute;
import com.bxm.adx.common.sell.builder.BuildAttributeStringMacrosHandler;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.openlog.extension.client.OpenLogClient;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import lombok.Data;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.util.UriComponentsBuilder;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author fgf
 * @date 2022/6/1
 **/
@Configuration
@Slf4j
public class DotDspBidEventListener implements EventListener<DspBidEvent> {

    private final int EXPIRE_SECONDS_12_HOURS = 12 * 3600;
    private final int EXPIRE_SECONDS_24_HOURS = 24 * 3600;

    private final Counter counter;
    private OpenLogClient openLogClient;
    private final AdxProperties properties;
    private final BuildAttributeStringMacrosHandler macrosHandler;
    private final AdxDomainMappingConfigCached adxDomainMappingConfigCached;

    public DotDspBidEventListener(Counter counter, AdxProperties properties, BuildAttributeStringMacrosHandler macrosHandler, AdxDomainMappingConfigCached adxDomainMappingConfigCached) {
        this.counter = counter;
        this.properties = properties;
        this.macrosHandler = macrosHandler;
        this.adxDomainMappingConfigCached = adxDomainMappingConfigCached;
    }

    @Autowired
    public void setOpenLogClient(OpenLogClient openLogClient) {
        this.openLogClient = openLogClient;
    }

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(DspBidEvent event) {
        handleDspAvgFill(event);
        if (Math.abs(event.getSspRequest().getId().hashCode() % 100) >= properties.getDotSimplingPercent()) {
            return;
        }
        MultiValueMap<String, String> customizeParams = buildCustomizeParams(event);
        String url = properties.getOpenLog().create(true, OpenlogConstants.Mt.DSP_RESPONSE_MT, customizeParams, OpenlogConstants.WITHOUT_MACROS_INNER);
        BuildAttribute attribute = new BuildAttribute(OpenlogConstants.Mt.DSP_RESPONSE_MT)
                .setDspId(event.getDspId())
                .setSspRequest(event.getSspRequest())
                .setAdxRequest(event.getAdxRequest())
                .setDspResponse(event.getDspResponse())
                .setStrategyId(event.getStrategyId());
        url = macrosHandler.replaceAll(url, attribute);
        openLogClient.asyncRequest(url,
                fallback -> log.warn("url = {}, err = {}", fallback.getUrl(), fallback.getException() == null ? "" : fallback.getException().getMessage()));
    }

    /**
     * 通过dsp单位时间内的填充数来控制是否请求dsp
     * @param event
     */
    private void handleDspAvgFill(DspBidEvent event) {
        String dspId = event.getDspId().toString();
        String dspTagId = event.getAdxRequest().getImps().iterator().next().getTag_id();

        LocalDateTime now = LocalDateTime.now();
        DotEventControlProperties control = needControl(dspId, dspTagId, now);
        if (Objects.isNull(control)) {
            return;
        }

        KeyGenerator sectionKey = AdxKeyGenerator.Counter.getDspPositionFillBySection(control.getId(), control.getSectionId(now),
                dspId, dspTagId);

        KeyGenerator controlKey = AdxKeyGenerator.Counter.getDspPositionFillByControl(control.getId(), dspId, dspTagId);

        counter.incrementAndGet(sectionKey, EXPIRE_SECONDS_12_HOURS);
        counter.incrementAndGet(controlKey, EXPIRE_SECONDS_24_HOURS);
    }

    private DotEventControlProperties needControl(String dspId, String dspTagId, LocalDateTime now) {
        List<DotEventControlProperties> controls = properties.getDotEventControls();
        if (CollectionUtils.isEmpty(controls)) {
            return null;
        }

        Optional<DotEventControlProperties> optional = controls.stream()
                .filter(p -> Objects.nonNull(p.getMt()) && OpenlogConstants.Mt.DSP_RESPONSE_MT == p.getMt())
                .filter(p -> p.getDspId().equals(dspId))
                .filter(p -> p.getDspPositionId().equals(dspTagId))
                .filter(p -> now.isAfter(p.getStart()) && now.isBefore(p.getEnd()))
                .findFirst();
        return optional.isPresent() ?  optional.get() : null;
    }

    /**
     * 构建客制化埋点参数
     */
    private MultiValueMap<String, String> buildCustomizeParams(DspBidEvent event) {
        MultiValueMap<String, String> customizeParams = new LinkedMultiValueMap<>();
        List<Creative> creatives = getCreatives(event.getDspResponse());
        String value = UrlHelper.urlEncode(JsonHelper.convert(creatives));
        customizeParams.add(OpenlogConstants.MacrosCustomizeParams.BID_CREATIVES, value);
        BidRequest request = event.getSspRequest();
        Device device = request.getDevice();
        if (Objects.nonNull(device)) {
            String verCodeOfHms = device.getVer_code_of_hms();
            if (StringUtils.isNotBlank(verCodeOfHms)) {
                customizeParams.add(OpenlogConstants.MacrosCustomizeParams.HMS_VER, verCodeOfHms);
            }
        }
        return customizeParams;
    }

    /**
     * 参竞广告详情
     * @param bidResponse
     * @return
     */
    private List<Creative> getCreatives(BidResponse bidResponse) {
        List<Creative> creatives = new ArrayList<>();
        for (SeatBid seatBid : bidResponse.getSeat_bid()) {
            for (Bid bid : seatBid.getBid()) {
                Creative creative = new Creative()
                        .setCreativeId(bid.getCreate_id())
                        .setTicketId(bid.getAdid())
                        .setBdgtyp(bid.getDsp_budget_type())
                        .setPrice(bid.getDsp_price().toString())
                        .setTid(bid.getTask_id())
                        .setDpcid(bid.getDpc_id())
                        .setDpcsid(bid.getDpc_strategy_id())
                        .setUsl(bid.getUser_score_level())
                        .setDid(adxDomainMappingConfigCached.getDomainId(bid))
                        ;
                creatives.add(creative);
            }
        }
        return creatives;
    }

    @Data
    @Accessors(chain = true)
    private static class Creative {
        private String creativeId;
        private String ticketId;
        private String price;
        private String tid;
        private String dpcid;
        private Integer bdgtyp;
        private Integer usl;
        private String dpcsid;
        private Long did;
    }
}
