package com.bxm.adx.common.openlog.listener.internal;

import com.alibaba.fastjson.JSON;
import com.bxm.adx.common.CacheKeys;
import com.bxm.adx.common.limiter.InsertExpireHandler;
import com.bxm.adx.common.openlog.event.internal.DspBidEvent;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.integration.pair.Pair;
import lombok.Data;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Configuration;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author fgf
 * @date 2024/7/22
 **/
@Configuration
public class DotDspBidForDeviceEventListener implements EventListener<DspBidEvent> {

    private final InsertExpireHandler insertExpireHandler;
    private final Pair pair;
    private final static String KEY = "bid.crowd.config";

    public DotDspBidForDeviceEventListener(@Qualifier("insertExpireDeviceCrowdPackageHandler") InsertExpireHandler insertExpireHandler, Pair pair) {
        this.insertExpireHandler = insertExpireHandler;
        this.pair = pair;
    }

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(DspBidEvent event) {
        String config = pair.get(KEY).of();
        if (StringUtils.isBlank(config)) {
            return;
        }
        List<Properties> configList = JSON.parseArray(config, Properties.class);
        if (CollectionUtils.isEmpty(configList)){
            return;
        }
        for (Properties properties : configList) {
            Long dspId = event.getDspId();
            if (dspId.equals(properties.dspId) && checkTaskId(properties, event) && checkDspTagId(properties, event)) {
                String crowdId = properties.getCrowdId();
                LocalDate now = LocalDate.now();
                if (now.isBefore(properties.getStart()) || now.isAfter(properties.getEnd())) {
                    return;
                }
                List<KeyGenerator> keys = CacheKeys.getCrowdPackageNewKeys(event.getSspRequest().getDevice());
                if (CollectionUtils.isNotEmpty(keys)) {
                    long crowdExpire = getExpire();
                    long keyExpire = crowdExpire - System.currentTimeMillis() / 1000;
                    keys.forEach(key -> {
                        insertExpireHandler.insert(key, crowdId, crowdExpire + "", (int) keyExpire);});
                }
            }
        }
    }

    /**
     * 检查任务ID
     * @param properties
     * @param event
     * @return
     */
    private boolean checkTaskId(Properties properties, DspBidEvent event) {
        String taskIds = properties.taskIds;
        if (StringUtils.isBlank(taskIds)) {
            return true;
        }
        List<String> bidTaskIds = event.getDspResponse().getSeat_bid().stream()
                .map(SeatBid::getBid)
                .flatMap(Collection::stream)
                .map(Bid::getTask_id)
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(bidTaskIds)) {
            return false;
        }
        List<String> taskIdList = properties.getTaskIds();
        for (String id : bidTaskIds) {
            if (taskIdList.contains(id)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 检查端点ID
     * @param properties
     * @param event
     * @return
     */
    private boolean checkDspTagId(Properties properties, DspBidEvent event) {
        String dspTagIds = properties.dspTagIds;
        if (StringUtils.isBlank(dspTagIds)) {
            return true;
        }
        String dspTagId = event.getAdxRequest().getImps().iterator().next().getTag_id();
        return properties.getDspTagIds().contains(dspTagId);
    }

    private long getExpire() {
        // 获取当前时间
        LocalDateTime now = LocalDateTime.now();
        // 计算过期的时间
        LocalDateTime oneWeekLater = now.plus(1, ChronoUnit.MONTHS);
        // 获取当前时区b
        ZoneId currentZone = ZoneId.systemDefault();
        // 将过期的时间转换为时间戳（毫秒级别）
        long timestamp = oneWeekLater.atZone(currentZone).toInstant().toEpochMilli() / 1000;
        return timestamp;
    }

    @Data
    private static class Properties {
        private Long dspId;
        private String crowdId;
        /**
         * 任务ID
         * 1,2,3.....
         */
        private String taskIds;
        /**
         * 端点id
         * 1,2,3...
         */
        private String dspTagIds;
        private LocalDate start;
        private LocalDate end;
        public Properties() {
        }

        public Properties(Long dspId, String crowdId, String taskIds, String dspTagIds, LocalDate start, LocalDate end) {
            this.dspId = dspId;
            this.crowdId = crowdId;
            this.taskIds = taskIds;
            this.dspTagIds = dspTagIds;
            this.start = start;
            this.end = end;
        }

        public List<String> getTaskIds() {
            if (StringUtils.isBlank(taskIds)) {
                return null;
            }
            return Arrays.asList(taskIds.split(","));
        }

        public List<String> getDspTagIds() {
            if (StringUtils.isBlank(dspTagIds)) {
                return null;
            }
            return Arrays.asList(dspTagIds.split(","));
        }
    }
}
