package com.bxm.adx.common.openlog.listener.internal;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.OpenlogConstants;
import com.bxm.adx.common.openlog.event.internal.DspBidEvent;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.builder.BuildAttribute;
import com.bxm.adx.common.sell.builder.BuildAttributeStringMacrosHandler;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.openlog.extension.client.OpenLogClient;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.integration.pair.Pair;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import lombok.Data;
import lombok.experimental.Accessors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.CollectionUtils;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 指定端点*广告位埋点3全量
 */
@Slf4j
@Configuration
public class DotDspBidForFullEventListener implements EventListener<DspBidEvent> {
    private OpenLogClient openLogClient;
    private final AdxProperties properties;
    private final BuildAttributeStringMacrosHandler macrosHandler;
    private final Pair pair;
    // 指定策略id埋点3全量
    private static String KEY = "dot.dsp.bid.full";

    public DotDspBidForFullEventListener(AdxProperties properties, BuildAttributeStringMacrosHandler macrosHandler, Pair pair) {
        this.properties = properties;
        this.macrosHandler = macrosHandler;
        this.pair = pair;
    }

    @Autowired
    public void setOpenLogClient(OpenLogClient openLogClient) {
        this.openLogClient = openLogClient;
    }

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(DspBidEvent event) {
        boolean full = isFull(event, pair);
        if (full) {
            MultiValueMap<String, String> customizeParams = buildCustomizeParams(event);
            String url = properties.getOpenLog().create(true, OpenlogConstants.Mt.DSP_BID_FULL, customizeParams, OpenlogConstants.WITHOUT_MACROS_INNER);
            BuildAttribute attribute = new BuildAttribute(OpenlogConstants.Mt.DSP_RESPONSE_MT)
                    .setDspId(event.getDspId())
                    .setSspRequest(event.getSspRequest())
                    .setAdxRequest(event.getAdxRequest())
                    .setDspResponse(event.getDspResponse())
                    .setStrategyId(event.getStrategyId());
            url = macrosHandler.replaceAll(url, attribute);
            openLogClient.asyncRequest(url,
                    fallback -> log.warn("url = {}, err = {}", fallback.getUrl(), fallback.getException() == null ? "" : fallback.getException().getMessage()));
        }
    }

    private boolean isFull(DspBidEvent event, Pair pair) {
        BidResponse dspResponse = event.getDspResponse();
        if (Objects.isNull(dspResponse)) {
            return false;
        }
        List<SeatBid> seatBids = dspResponse.getSeat_bid();
        if (CollectionUtils.isEmpty(seatBids)) {
            return false;
        }
        Set<String> strategyIds = pair.get(KEY).ofHashSet();
        if (CollectionUtils.isEmpty(strategyIds)){
            return false;
        }
        for (SeatBid seatBid : seatBids) {
            List<Bid> bids = seatBid.getBid();
            if (!CollectionUtils.isEmpty(bids)) {
                for (Bid bid : bids) {
                    if (Objects.nonNull(bid) && StringUtils.isNotBlank(bid.getDpc_id())){
                        if (strategyIds.contains(bid.getDpc_id())) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }

    /**
     * 构建客制化埋点参数
     */
    private MultiValueMap<String, String> buildCustomizeParams(DspBidEvent event) {
        MultiValueMap<String, String> customizeParams = new LinkedMultiValueMap<>();
        List<DotDspBidForFullEventListener.Creative> creatives = getCreatives(event.getDspResponse());
        String value = UrlHelper.urlEncode(JsonHelper.convert(creatives));
        customizeParams.add(OpenlogConstants.MacrosCustomizeParams.BID_CREATIVES, value);
        BidRequest request = event.getSspRequest();
        Device device = request.getDevice();
        if (Objects.nonNull(device)) {
            String verCodeOfHms = device.getVer_code_of_hms();
            if (StringUtils.isNotBlank(verCodeOfHms)) {
                customizeParams.add(OpenlogConstants.MacrosCustomizeParams.HMS_VER, verCodeOfHms);
            }
        }
        return customizeParams;
    }

    /**
     * 参竞广告详情
     * @param bidResponse
     * @return
     */
    private List<DotDspBidForFullEventListener.Creative> getCreatives(BidResponse bidResponse) {
        List<DotDspBidForFullEventListener.Creative> creatives = new ArrayList<>();
        for (SeatBid seatBid : bidResponse.getSeat_bid()) {
            for (Bid bid : seatBid.getBid()) {
                DotDspBidForFullEventListener.Creative creative = new DotDspBidForFullEventListener.Creative()
                        .setCreativeId(bid.getCreate_id())
                        .setTicketId(bid.getAdid())
                        .setBdgtyp(bid.getDsp_budget_type())
                        .setPrice(bid.getDsp_price().toString())
                        .setTid(bid.getTask_id())
                        .setDpcid(bid.getDpc_id())
                        .setDpcsid(bid.getDpc_strategy_id())
                        .setUsl(bid.getUser_score_level());
                creatives.add(creative);
            }
        }
        return creatives;
    }

    @Data
    @Accessors(chain = true)
    private static class Creative {
        private String creativeId;
        private String ticketId;
        private String price;
        private String tid;
        private String dpcid;
        private Integer bdgtyp;
        private Integer usl;
        private String dpcsid;
    }
}
