package com.bxm.adx.common.report;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.CacheKeys;
import com.bxm.adx.common.utils.DateUtils;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.utils.NamedThreadFactory;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.RandomUtils;
import org.apache.curator.shaded.com.google.common.collect.Lists;
import org.apache.http.message.BasicHeader;
import org.springframework.scheduling.annotation.Scheduled;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Set;
import java.util.concurrent.*;
import java.util.function.Consumer;

/**
 * @author fgf
 * @date 2025/5/8
 **/
@Slf4j
public class AttributionReportService {
    private final Updater updater;
    private final Fetcher fetcher;
    private final JedisPool jedisPool;
    private final ReportClient reportClient;
    private final ThreadPoolExecutor executor = new ThreadPoolExecutor(1, 1, 0, TimeUnit.SECONDS,
            new SynchronousQueue<>(), new NamedThreadFactory("attribution-report"), new ThreadPoolExecutor.DiscardPolicy());
    public final static List<Long> SUPPORT_DSP = Lists.newArrayList(
            97L
    );
    public final static List<Integer> SUPPORT_MEDIA = Lists.newArrayList(
            AdxConstants.Media.Bes.getId()
    );
    public final static int EXPIRE = 24 * 3600;
    public final static int DB = 4;

    public AttributionReportService(Updater updater, Fetcher fetcher, JedisPool jedisPool, ReportClient reportClient) {
        this.updater = updater;
        this.fetcher = fetcher;
        this.jedisPool = jedisPool;
        this.reportClient = reportClient;
    }

    /**
     * 保存有归因可能的点击url
     *
     * @param bidId
     * @param url
     */
    public void saveClickUrl(String bidId, String url) {
        String date = LocalDate.now().format(DateTimeFormatter.ofPattern(DateUtils.PATTERN_DATE));
        updater.updateWithSelector(CacheKeys.ServerReport.getAttributionKey(date, bidId), url, EXPIRE, DB);
    }

    /**
     * 定时任务执行上报点击
     */
//    @DisableDistributed
    @Scheduled(cron = "43 17 * * * ?")
    public void reportByBidId() {
        long begin = System.currentTimeMillis();
        Future<Integer> future = executor.submit(new Callable<Integer>() {
            @Override
            public Integer call() throws Exception {
                log.info("attribution report start");
                String date = LocalDate.now().format(DateTimeFormatter.ofPattern(DateUtils.PATTERN_DATE));
                int num = 0;
//                List<String> bidIds = getBidIds(date);
                Set<String> bidIds = getBidIdsByData(date);
                if (CollectionUtils.isEmpty(bidIds)) {
                    return num;
                }
                log.info("attribution report bidIds size = {}", bidIds.size());
                for (String id : bidIds) {
                    if (StringUtils.isBlank(id)) {
                        continue;
                    }
                    log.debug("id = {}", id);
                    KeyGenerator key = CacheKeys.ServerReport.getAttributionKey(date, id);
                    String url = fetcher.fetchWithSelector(key, String.class, DB);
                    if (StringUtils.isNotBlank(url)) {
                        reportClient.asyncRequest(url, new Consumer<ReportFallback>() {
                            @Override
                            public void accept(ReportFallback reportFallback) {
                                Exception exception = reportFallback.getException();
                                log.warn("attribution report fail {} url {}, headers {}",
                                        exception == null ? "" : exception.getMessage(), reportFallback.getUrl(), reportFallback.getHeaders());
                            }
                        }, new BasicHeader[0]);
                        num++;
                        Thread.sleep(RandomUtils.nextInt(10, 3000));
                    }

                }
                return num;
            }
        });
        try {
            log.info("attribution report end, cost = {} total = {}", (System.currentTimeMillis() - begin) / 1000, future.get());
        } catch (Exception e) {
            log.error("err", e);
        }
    }

    /**
     * 获取有回传的bidid
     *
     * @param date
     * @return
     */
    private List<String> getBidIds(String date) {
        String key = CacheKeys.ServerReport.getAttributionBidKey(date).generateKey();
        try (Jedis jedis = jedisPool.getResource()) {
            jedis.select(DB);
            return jedis.lrange(key, 0, -1);
        }
    }

    /**
     * 获取大数据写入的bes有回传的bidid
     *
     * @param date
     * @return
     */
    private Set<String> getBidIdsByData(String date) {
        String key = CacheKeys.ServerReport.getAttributionBidByDataKey(date).generateKey();
        try (Jedis jedis = jedisPool.getResource()) {
            jedis.select(DB);
            return jedis.smembers(key);
        }
    }
}
