package com.bxm.adx.common.rule;

import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

/**
 * 支持黑白名单的 Set 格式的规则，将 {@link com.bxm.mccms.facade.model.Rule#getRuleValue()} 拆分为数组然后放入 Set 集合中。
 * 可通过 {@link #isWhite()} 来判别黑白。
 *
 * @author allen
 * @date 2021-06-11
 * @since 1.0
 */
@Slf4j
public class WhiteBlackSetRule {

    public static final String TYPE_WHITELIST = "1";
    public static final String TYPE_BLACKLIST = "2";

    private boolean white;

    private final static String GROUP_SPLIT = "-";
    private final static String SPLIT = ",";
    private final Rule rule;
    private final String groupSeparatorChars;
    private final String separatorChars;

    /**
     * 构造方法，默认使用 {@link #SPLIT} 进行分隔。
     *
     * @param rule Rule对象
     */
    public WhiteBlackSetRule(Rule rule) {
        this(rule, GROUP_SPLIT, SPLIT);
    }

    /**
     * 构造方法
     *
     * @param rule Rule对象
     * @param groupSeparatorChars 用作分隔黑白标识的字符
     * @param separatorChars 用作分隔符的字符
     */
    public WhiteBlackSetRule(Rule rule, String groupSeparatorChars, String separatorChars) {
        this.rule = rule;
        this.groupSeparatorChars = groupSeparatorChars;
        this.separatorChars = separatorChars;
    }

    /**
     * 返回拆分后的集合
     *
     * @return 拆分后的集合
     */
    public Set<String> getSet() {
        String value = rule.getRuleValue();
        HashSet<String> res = Sets.newHashSet();
        if (StringUtils.isBlank(value)) {
            return res;
        }
        int startIndex = value.indexOf(groupSeparatorChars);
        if (startIndex == -1) {
            log.warn("Invalid value for value: {}", value);
            return res;
        }

        String type = StringUtils.substring(value, 0, startIndex);
        String array = StringUtils.substring(value, startIndex + 1, value.length());

        this.white = StringUtils.equalsIgnoreCase(type, TYPE_WHITELIST);
        return Sets.newHashSet(StringUtils.split(array, separatorChars));
    }

    /**
     * <p>返回这个集合是否白名单</p>
     * <p>注意：需要先调用 {@link #getSet()} 之后才能确认黑白名单，请务必保证 {@link #isWhite()} 在 {@link #getSet()} 之后访问。</p>
     * @return 是，白名单。
     */
    public boolean isWhite() {
        return this.white;
    }

    /**
     * <p>是否需要过滤</p>
     * <p>统计 {@code basic} 集合元素在 {@code rule} 元素匹配的数量，然后根据黑白名单类型来返回结果。</p>
     *
     * <p>示例：</p>
     * <li>黑名单集合 {@code rule} 的元素是 0,1,2，当 {@code basic} 集合元素是 0,1时，返回 true</li>
     * <li>黑名单集合 {@code rule} 的元素是 0,1,2，当 {@code basic} 集合元素是 3,4时，返回 false</li>
     * <li>白名单集合 {@code rule} 的元素是 0,1,2，当 {@code basic} 集合元素是 0,1时，返回 false</li>
     * <li>白名单集合 {@code rule} 的元素是 0,1,2，当 {@code basic} 集合元素是 3,4时，返回 true</li>
     *
     *
     * @param rule 黑白名单规则模型
     * @param basic 匹配列表
     * @return
     * <p>白名单，如果匹配为 0，则返回 true。</p>
     * <p>黑名单，如果匹配大于 0，则返回 true。</p>
     */
    public static boolean isFilter(Rule rule, Collection<String> basic) {
        WhiteBlackSetRule wbsr = new WhiteBlackSetRule(rule);
        Set<String> defined = wbsr.getSet();
        boolean white = wbsr.isWhite();

        long count = defined.stream().filter(basic::contains).count();
        if (white) {
            return count == 0;
        } else {
            return count > 0;
        }
    }
}
