package com.bxm.adx.common.sell.builder;

import com.bxm.adx.common.sell.builder.macros.StringMacrosHandler;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Configuration;

import java.util.*;

/**
 * 字符串型宏参替换处理器，基于BuildAttribute属性替换。
 *
 * @author allen
 * @date 2021-06-22
 * @since 1.0
 */
@Configuration
public class BuildAttributeStringMacrosHandler implements StringMacrosHandler<BuildAttribute>, ApplicationContextAware {

    private final List<BuildAttributeStringMacros> macros = Lists.newArrayList();

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        Map<String, BuildAttributeStringMacros> beans = applicationContext.getBeansOfType(BuildAttributeStringMacros.class);
        macros.addAll(beans.values());
    }

    @Override
    public String replaceAll(String string, BuildAttribute attribute) {
        return replaceAll(string, attribute, new String[0]);
    }

    @Override
    public String replaceAll(String string, BuildAttribute attribute, String... excludes) {
        if (Objects.isNull(attribute)) {
            return string;
        }
        if (StringUtils.isBlank(string)) {
            return string;
        }
        Set<String> ex = new HashSet<>();
        if (ArrayUtils.isNotEmpty(excludes)) {
            ex = Sets.newHashSet(excludes);
        }
        // 宏参替换
        for (BuildAttributeStringMacros macro : macros) {
            String macroString = macro.getMacroString();
            if (ex.contains(macroString)) {
                continue;
            }
            String replacement;
            try {
                replacement = macro.apply(attribute);
            } catch (Exception e) {
                replacement = macro.getIfBlankValue();
            }
            if (Objects.isNull(replacement)) {
                replacement = macro.getIfBlankValue();
            }
            string = string.replaceAll(macroString, replacement);
        }

        return string;
    }
}
