package com.bxm.adx.common.utils;

import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.util.concurrent.MoreExecutors;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.binder.MeterBinder;
import lombok.extern.slf4j.Slf4j;

import java.util.Objects;
import java.util.concurrent.*;
import java.util.function.Consumer;

/**
 * 抽象异步（线程池）的接受单个参数且不返回结果的操作。
 *
 * @author allen
 * @date 2021-07-19
 * @since 1.0
 */
@Slf4j
public abstract class AbstractSyncProcessor<T> implements MeterBinder, Consumer<T> {

    private final ExecutorService executorService;
    private final LinkedBlockingQueue<Runnable> queue;

    public AbstractSyncProcessor(int coreSize, int maximumLength, String threadName, RejectedExecutionHandler rejectedExecutionHandler) {
        this.queue = new LinkedBlockingQueue<>(maximumLength);
        ThreadPoolExecutor threadPoolExecutor = new ThreadPoolExecutor(coreSize, coreSize, 0, TimeUnit.SECONDS, queue, new NamedThreadFactory(threadName), rejectedExecutionHandler);
        this.executorService = MoreExecutors.getExitingExecutorService(threadPoolExecutor);
    }

    /**
     * @return 返回 Gauge 统计的名称，请保证在应用内是唯一的。
     */
    protected abstract String getGaugeName();

    /**
     * @return 返回处理 T 对象的具体实现。
     */
    protected abstract Consumer<T> getConsumer();

    @Override
    public void bindTo(MeterRegistry registry) {
        Gauge.builder(getGaugeName() + ".queue.blocks", 0, (value -> queue.size())).register(registry);
    }

    @Override
    public void accept(T t) {
        Consumer<T> consumer = getConsumer();
        if (Objects.isNull(consumer)) {
            return;
        }
        this.executorService.submit(() -> {
            try {
                consumer.accept(t);
            } catch (Exception e) {
                log.warn("accept: {}", e.getMessage());
            }
        });
    }
}
