package com.bxm.adx.common.autoconfigure;

import com.bxm.adx.common.adapter.BidModelAdapter;
import com.bxm.adx.common.adapter.aspect.ModelAdapterRequestLogAdvice;
import com.bxm.adx.common.log.AdxDataLogMonitor;
import com.bxm.adx.common.log.ByteLogger;
import com.bxm.adx.common.log.buriedSwitch.BuriedSwitchCached;
import com.bxm.adx.common.sell.position.mapping.PositionMappingStrategyFactory;
import com.bxm.warcar.dpl2.plugin.spring.BeforeRefreshContextHandler;
import lombok.extern.slf4j.Slf4j;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.aop.support.DefaultPointcutAdvisor;
import org.springframework.aop.support.NameMatchMethodPointcut;
import org.springframework.aop.support.RootClassFilter;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.AnnotationConfigApplicationContext;
import org.springframework.context.annotation.Bean;

/**
 * @author fgf
 * @date 2024/3/4
 **/
@Slf4j
@ConditionalOnProperty(prefix = "adx", name = "enable-model-adapter-proxy-factory", matchIfMissing = false, havingValue = "true")
public class ModelAdapterProxyFactoryAutoConfiguration implements BeforeRefreshContextHandler {

    private final ByteLogger byteLogger;
    private final PositionMappingStrategyFactory positionMappingStrategyFactory;
    private final BuriedSwitchCached buriedSwitchCached;
    private final AdxDataLogMonitor adxDataLogMonitor;
    public ModelAdapterProxyFactoryAutoConfiguration(ApplicationContext applicationContext) {
        this.byteLogger = applicationContext.getBean(ByteLogger.class);
        this.positionMappingStrategyFactory = applicationContext.getBean(PositionMappingStrategyFactory.class);
        this.buriedSwitchCached = applicationContext.getBean(BuriedSwitchCached.class);
        this.adxDataLogMonitor = applicationContext.getBean(AdxDataLogMonitor.class);
    }

    @Bean
    public ModelAdapterRequestLogAdvice modelAdapterAdvice() {
        return new ModelAdapterRequestLogAdvice(byteLogger, positionMappingStrategyFactory, buriedSwitchCached, adxDataLogMonitor);
    }

    @Bean
    public NameMatchMethodPointcut modelAdapterPointcut() {
        NameMatchMethodPointcut pointcut = new NameMatchMethodPointcut();
        pointcut.setClassFilter(new RootClassFilter(BidModelAdapter.class));
        pointcut.setMappedName("convert");
        return pointcut;
    }

    @Bean
    public DefaultPointcutAdvisor modelAdapterAdvisor() {
        return new DefaultPointcutAdvisor(modelAdapterPointcut(), modelAdapterAdvice());
    }

    @Override
    public void accept(AnnotationConfigApplicationContext annotationConfigApplicationContext) {
        ConfigurableListableBeanFactory beanFactory = annotationConfigApplicationContext.getBeanFactory();
        beanFactory.addBeanPostProcessor(new BeanPostProcessor() {
            @Override
            public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
                if (bean instanceof BidModelAdapter) {
                    ProxyFactory factory = new ProxyFactory();
                    factory.setTarget(bean);
                    factory.addAdvisor(modelAdapterAdvisor());
                    return factory.getProxy();
                }
                return bean;
            }
        });
    }
}
