package com.bxm.adx.common.buy.dispatcher.filter;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherContext;
import com.bxm.adx.common.caching.Id;
import com.bxm.adx.common.rule.Rule;
import com.bxm.adx.common.rule.AndOrSetRule;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.warcar.integration.pair.Pair;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.annotation.Configuration;

import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author fgf
 * @date 2023/3/13
 **/
@DispatcherFilterCondition(probeOn = false)
@Slf4j
@Configuration
public class DeviceFilter implements DispatcherFilter<Dispatcher> {

    private final Pair pair;

    private static final Set<String> DEVICE_ID_NAMES = Sets.newHashSet("imei", "idfa", "oaid", "gaid");
    /**
     * 需过滤的异常设备号key
     */
    private static final String EXCEPTION_DEVICE_LIST = "adx.exception.device.list";

    public DeviceFilter(Pair pair) {
        this.pair = pair;
    }

    @Override
    public void filter(DispatcherContext<Dispatcher> context, Set<Id> trash) {
        Device device = context.getRequest().getDevice();
        filterDevice(context.getRequest(), device);
        Set<Dispatcher> removes = context.getValues().stream().filter(dispatcher -> limitByDevice(dispatcher, device))
                .collect(Collectors.toSet());
        if (CollectionUtils.isNotEmpty(removes)) {
            trash.addAll(removes);
        }
    }

    /**
     * 如果命中实验组，过滤无效设备号
     */
    private void filterDevice(BidRequest request, Device device) {
        if (request.getExpId() == AdxConstants.NO) {
            return;
        }
        Set<String> blackSet = Optional.of(pair.get(EXCEPTION_DEVICE_LIST).ofHashSet()).orElse(new HashSet<>());

        //imei不符合14或者15位数字、imei_md5/idfa_md5/oaid_md5不符合32位无效
        String imei = device.getImei();

        String imeiMd5 = device.getImei_md5();
        String oaidMd5 = device.getOaid_md5();
        String idfaMd5 = device.getIdfa_md5();
        if (StringUtils.isNotBlank(imei) &&
                (!imei.matches("\\d+")
                        || (imei.length() != 14 && imei.length() != 15)
                        || (blackSet.contains(imei)))) {
            device.setImei(null);
            if (DigestUtils.md5Hex(imei).equals(imeiMd5)) {
                device.setImei_md5(null);
            }
        }

        String idfa = device.getIdfa();
        if (StringUtils.isNotBlank(idfa) && (blackSet.contains(idfa))) {
            device.setIdfa(null);
            if (DigestUtils.md5Hex(idfa).equals(idfaMd5)) {
                device.setIdfa_md5(null);
            }
        }

        String oaid = device.getOaid();
        if (StringUtils.isNotBlank(oaid) && (blackSet.contains(oaid))) {
            device.setOaid(null);
            if (DigestUtils.md5Hex(oaid).equals(oaidMd5)) {
                device.setOaid_md5(null);
            }
        }

        if (isErrorDevice(imeiMd5, blackSet)) {
            device.setImei_md5(null);
        }

        if (isErrorDevice(idfaMd5, blackSet)) {
            device.setIdfa_md5(null);
        }

        if (isErrorDevice(oaidMd5, blackSet)) {
            device.setOaid_md5(null);
        }
    }

    /**
     * 判断MD5是不是异常
     *
     * @param md5      md5
     * @param blackSet 黑名单
     * @return true 异常
     */
    private boolean isErrorDevice(String md5, Set<String> blackSet) {
        return (StringUtils.isNotBlank(md5) && md5.length() != 32) || blackSet.contains(md5);
    }

    @Override
    public int getOrder() {
        return 1;
    }

    private boolean limitByDevice(Dispatcher dispatcher, Device device) {
        Rule deviceRule = dispatcher.getDeviceInfo();
        if (Objects.isNull(deviceRule)) {
            return false;
        }
        String ruleVal = deviceRule.getRuleValue();
        if (StringUtils.isNotEmpty(ruleVal)) {
            deviceRule.setRuleValue(ruleVal.replace(",", "/"));
        }
        AndOrSetRule andOrSetRule = new AndOrSetRule(deviceRule);
        Set<String> paramNames = andOrSetRule.getAndParamNameSet();
        for (String paramName : paramNames) {
            if (andOrSetRule.isOrParams(paramName)) {
                Set<String> orParams = andOrSetRule.getOrParamNameSet(paramName);
                if (CollectionUtils.isEmpty(orParams)) {
                    continue;
                }
                boolean orLimit = false;
                for (String orParamName : orParams) {
                    if (hasFieldVal(device, orParamName)) {
                        orLimit = true;
                        break;
                    }
                }
                if (!orLimit) {
                    return true;
                }
            } else {
                if (!hasFieldVal(device, paramName)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * 反射获取属性名对应的value
     * 如果是设备ID相关字段还需要查询md5的属性值
     *
     * @param device
     * @param fieldName
     * @return
     */
    private boolean hasFieldVal(Device device, String fieldName) {
        List<String> fieldNames = new ArrayList<>();
        fieldNames.add(fieldName);
        if (DEVICE_ID_NAMES.contains(fieldName)) {
            fieldNames.add(fieldName + "_md5");
        }
        try {
            Class cla = device.getClass();
            for (String name : fieldNames) {
                Field field = cla.getDeclaredField(name);
                field.setAccessible(true);
                Object val = field.get(device);
                if (log.isDebugEnabled()) {
                    log.debug("check device param {} val {}", name, Optional.ofNullable(val).orElse(null));
                }
                boolean hasVal = Objects.nonNull(val) && StringUtils.isNotBlank(val.toString());
                if (hasVal) {
                    return true;
                }
            }
        } catch (Exception e) {
        }
        return false;
    }
}
