package com.bxm.adx.common.buy.dispatcher.filter;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherContext;
import com.bxm.adx.common.caching.Id;
import com.bxm.adx.common.sell.request.Impression;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author fgf
 * @date 2023/3/13
 **/
@Slf4j
@Configuration
@DispatcherFilterCondition(onDmType = AdxConstants.DockingMethodType.SDK)
public class PriceFilter implements DispatcherFilter<Dispatcher> {
    @Override
    public void filter(DispatcherContext<Dispatcher> context, Set<Id> trash) {
        if (!context.getSdkConcurrentModel()) {
            return;
        }
        Impression impression = context.getRequest().getImps().iterator().next();
        BigDecimal minPrice = impression.getBid_floor();
        BigDecimal maxPrice = impression.getBid_top();
        Collection<Dispatcher> dispatchers = context.getValues();
        Set<Dispatcher> removes = dispatchers.stream().filter(
                dispatcher -> !limitedByPrice(dispatcher, minPrice, maxPrice)
        ).collect(Collectors.toSet());
        if (CollectionUtils.isNotEmpty(removes)) {
            trash.addAll(removes);
        }
    }

    @Override
    public int getOrder() {
        return 0;
    }

    /**
     * 平均cpm价格过滤
     *
     * @param dispatcher
     * @param min
     * @param max
     * @return
     */
    private boolean limitedByPrice(Dispatcher dispatcher, BigDecimal min, BigDecimal max) {
        BigDecimal price = dispatcher.getDspAvgPrice();
        if (Objects.isNull(price)) {
            return true;
        }
        if (price.compareTo(min.movePointLeft(2)) == -1) {
            if (log.isDebugEnabled()) {
                log.debug("rm dispatcher {} by : {}", dispatcher.getId(), "< min");
            }
            return true;
        }
        if (price.compareTo(max.movePointLeft(2)) >= 0) {
            if (log.isDebugEnabled()) {
                log.debug("rm dispatcher {} by : {}", dispatcher.getId(), "> max");
            }
            return true;
        }
        return false;
    }
}
