package com.bxm.adx.common.market.exchange.rebuild.filter;

import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.market.exchange.rebuild.response.ResponseBuildAttribute;
import com.bxm.adx.common.openlog.event.internal.AdxDispatcherPriceConfigFilter;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author fgf
 * @date 2024/6/6
 **/
@Slf4j
@Configuration
public class DispatcherPriceConfigFilterFactory implements ApplicationListener<ApplicationReadyEvent> {
    private final Collection<DispatcherPriceConfigFilter> filters = Lists.newArrayList();
    private final EventPark eventPark;

    public DispatcherPriceConfigFilterFactory(EventPark eventPark) {
        this.eventPark = eventPark;
    }

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        ConfigurableApplicationContext context = event.getApplicationContext();
        Map<String, DispatcherPriceConfigFilter> filterMap = context.getBeansOfType(DispatcherPriceConfigFilter.class);
        if (CollectionUtils.isEmpty(filterMap)) {
            return;
        }
        filters.addAll(filterMap.values().stream().sorted(Comparator.comparing(Ordered::getOrder)).collect(Collectors.toList()));
        if (log.isInfoEnabled()) {
            filters.forEach(filter -> {
                log.info("Registered DispatcherPriceConfigFilter: {} - {}", filter.getOrder(), filter);
            });
        }
    }

    public void filter(BidResponse response, ResponseBuildAttribute attribute) {
        Dispatcher dispatcher = attribute.getDispatcher();
        List<DispatcherPriceConfig> priceConfigs = dispatcher.getDispatcherPriceConfigs();
        List<FilterInfo> filterInfos = new ArrayList<>();
        for (SeatBid seatBid : response.getSeat_bid()) {
            for (Bid bid : seatBid.getBid()) {
                Collection<DispatcherPriceConfig> collection = Lists.newArrayList(priceConfigs);
                Map<Integer, Set<Long>> filterConfigMap = Maps.newHashMap();

                filters.forEach(
                        filter -> {
                            Collection<DispatcherPriceConfig> temp = Lists.newArrayList(collection);
                            //过滤
                            filter.filter(collection, bid, seatBid, attribute);
                            //记录过滤掉的配置
                            Set<Long> filterConfigIds = diff(temp, collection);
                            if (!CollectionUtils.isEmpty(filterConfigIds)) {
                                filterConfigMap.put(DispatcherPriceConfigFilterEnum.getTypeByClass(filter), filterConfigIds);
                            }
                        }
                );

                if (collection.size() != 1) {
                    log.warn("dispatcher {} price config size is {}", dispatcher.getId(), collection.size());
                    throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
                }
                Long dpcId = collection.iterator().next().getPriceModeId();
                bid.setDpc_id(dpcId == null ? null : dpcId.toString());

                filterInfos.add(FilterInfo.builder()
                                .bid(bid)
                                .seatBid(seatBid)
                                .filterConfigMap(filterConfigMap)
                        .build());
            }
        }
        //埋点价格配置过滤情况
        if (!CollectionUtils.isEmpty(filterInfos)) {
            eventPark.post(new AdxDispatcherPriceConfigFilter(this, attribute.getBidRequest(), dispatcher,
                    attribute.getAdxRequest(), attribute.getBuyer(), filterInfos));
        }
    }

    /**
     * 筛查每次
     * @param before
     * @param after
     * @return
     */
    public Set<Long> diff(Collection<DispatcherPriceConfig> before, Collection<DispatcherPriceConfig> after) {
        return before.stream().filter(e -> !after.contains(e)).map(DispatcherPriceConfig::getPriceModeId).collect(Collectors.toSet());
    }
}
