package com.bxm.adx.common.market.exchange.rebuild.request;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.position.AdvertPoint;
import com.bxm.adx.common.buy.position.AdvertPointService;
import com.bxm.adx.common.buy.position.Constants;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.request.Impression;
import com.bxm.adx.common.sell.request.Native;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.mcssp.common.enums.app.DockingMethodTypeEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * * 重构BidRequest中的Impression
 *
 * @author fgf
 * @date 2023/1/6
 **/
@Slf4j
@Configuration
public class ImpressionBuilder implements AdxBidRequestBuilder {
    private final AdvertPointService advertPointService;
    private final static String DSP_QIHANG = "qihang";
    private final static String DSP_PANGU = "pangu";
    private final static Integer PANGU_CTR = 15;

    public ImpressionBuilder(AdvertPointService advertPointService) {
        this.advertPointService = advertPointService;
    }

    @Override
    public void rebuildAdxBidRequest(BidRequest adxBidRequest, RequestBuildAttribute attribute) {
        Buyer buyer = attribute.getBuyerWrapper().getBuyer();
        Dispatcher dispatcher = attribute.getBuyerWrapper().getDispatcher();
        AdvertPoint advertPoint = advertPointService.getAdvertPointByDspId(dispatcher.getDspId().toString(),
                dispatcher.getDspAppid(), dispatcher.getDspPosid());

        List<Impression> oldImpressions = adxBidRequest.getImps();
        List<Impression> newImpressions = new ArrayList<>();
        for (Impression impression : oldImpressions) {
            //init newImpression
            Impression newImpression = new Impression();
            BeanUtils.copyProperties(impression, newImpression);

            //替换DSP广告位ID
            if (needReplaceTagIdByDsp(adxBidRequest, buyer)) {
                newImpression.setTag_id(dispatcher.getDspPosid());
            }
            //根据Dsp-广告端点替换
            if (needReplaceByAdvertPoint(adxBidRequest)) {
                handleAdvertPoint(advertPoint, newImpression, buyer);
            }
            //底价处理
            handleBidFloor(newImpression, dispatcher, advertPoint, buyer);
            newImpressions.add(newImpression);
            break;
        }
        adxBidRequest.setImps(newImpressions);
    }

    /**
     * 处理Dsp-广告位端点
     *
     * @param advertPoint
     * @param newImpression
     * @param buyer
     */
    private void handleAdvertPoint(AdvertPoint advertPoint, Impression newImpression, Buyer buyer) {
        if (Objects.nonNull(advertPoint)) {
            //替换广告位尺寸
            AdvertPoint.Size size = advertPoint.getImpressionSize();
            if (Objects.nonNull(size)) {
                newImpression.setH(size.getH());
                newImpression.setW(size.getW());
            }
            //素材模版id
            if (StringUtils.isNotEmpty(advertPoint.getTemplateId())) {
                Native an = new Native();
                if (Objects.nonNull(newImpression.getA_native())) {
                    BeanUtils.copyProperties(newImpression.getA_native(), an);
                }
                an.setTemplate(advertPoint.getTemplateId());
                newImpression.setA_native(an);
            }
            //启航处理广告位类型
            Integer positionType = advertPoint.getPositionType();
            if (Objects.nonNull(positionType)) {
                if (DSP_QIHANG.equalsIgnoreCase(buyer.getCode())) {
                    Integer newImpType = Constants.getPositionSceneTypeByPositionType(positionType);
                    if (Objects.nonNull(newImpType)) {
                        newImpression.setImp_type(newImpType);
                    }
                }
            }
        }
    }

    /**
     * 底价处理
     *
     * @param newImpression
     * @param dispatcher
     */
    private void handleBidFloor(Impression newImpression, Dispatcher dispatcher, AdvertPoint advertPoint, Buyer buyer) {
        if (Objects.isNull(advertPoint)) {
            log.warn("dispatcher id: {}, advertPoint is null", dispatcher.getId());
            throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
        }
        //优先使用流量分配中的底价
        boolean isPangu = isPangu(buyer);
        BigDecimal dspRequestFloor = dispatcher.getDspBasePrice();
        Integer costPerType = Optional.ofNullable(advertPoint.getCostPerType()).orElse(AdxConstants.ChargeType.CPM);
        if (Objects.isNull(dspRequestFloor)) {
            //底价系数处理
            if (AdxConstants.ChargeType.CPC == costPerType) {
                log.warn("Dsp position is cpc, Dut dspBasePrice is null");
                throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
            }
            BigDecimal coefficient = dispatcher.getBasePriceCoefficient();
            if (Objects.nonNull(coefficient) && coefficient.compareTo(BigDecimal.ZERO) > 0) {
                dspRequestFloor = newImpression.getBid_floor().divide(coefficient, 0, BigDecimal.ROUND_UP);
            } else {
                dspRequestFloor = newImpression.getBid_floor();
            }

        } else {
            //转成分
            dspRequestFloor = dspRequestFloor.movePointRight(2);
        }
        //底价单位为分，不保留小数，有小数自动向上进位
        dspRequestFloor = dspRequestFloor.setScale(0, BigDecimal.ROUND_UP);
        switch (costPerType) {
            case AdxConstants.ChargeType.CPC:
                newImpression.setCpc_bid_floor(dspRequestFloor);
                if (!isPangu) {
                    newImpression.setBid_floor(null);
                }
                break;
            case AdxConstants.ChargeType.CPM:
                newImpression.setBid_floor(dspRequestFloor);
                newImpression.setCpc_bid_floor(null);
                break;
        }

        if (isPangu) {
            handlePanguBidFloor(newImpression);
        }
    }

    private boolean isPangu(Buyer buyer) {
        return buyer.getCode().contains(DSP_PANGU);
    }

    /**
     * 特殊处理盘古底价，盘古只支持cpm底价
     * 使用公式换算cpm = cpc * 固定ctr=15% * 1000
     *
     * @param newImpression
     */
    private void handlePanguBidFloor(Impression newImpression) {
        if (Objects.isNull(newImpression.getBid_floor()) && Objects.nonNull(newImpression.getCpc_bid_floor())) {
            BigDecimal floor = newImpression.getCpc_bid_floor()
                    .multiply(new BigDecimal(PANGU_CTR)).movePointLeft(2)
                    .multiply(new BigDecimal(1000));
            newImpression.setBid_floor(floor);
        }
    }

    /**
     * 是否需要替换tagId
     *
     * @param request
     * @param buyer
     * @return
     */
    private boolean needReplaceTagIdByDsp(BidRequest request, Buyer buyer) {
        DockingMethodTypeEnum typeEnum = dockingMethodTypeEnum(request);
        if (Objects.isNull(typeEnum)) {
            return true;
        }
        switch (typeEnum) {
            case SDK_OPERATION:
                if ("scene".equalsIgnoreCase(buyer.getCode())) {
                    return false;
                }
                return true;
            default:
                return true;
        }
    }

    /**
     * 是否需要根据Dsp-广告端点信息替换Impression
     *
     * @param request
     * @return
     */
    private boolean needReplaceByAdvertPoint(BidRequest request) {
        DockingMethodTypeEnum typeEnum = dockingMethodTypeEnum(request);
        if (DockingMethodTypeEnum.SDK_OPERATION.equals(typeEnum)) {
            return false;
        }
        return true;
    }

    /**
     * 广告位对接类型
     *
     * @param request
     * @return
     */
    private DockingMethodTypeEnum dockingMethodTypeEnum(BidRequest request) {
        Integer dockingMethodType = request.getDockingMethodType();
        DockingMethodTypeEnum typeEnum = DockingMethodTypeEnum.get(dockingMethodType);
        return typeEnum;
    }
}
