package com.bxm.adx.common.market.exchange.rebuild.response;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.creative.review.ReviewCreative;
import com.bxm.adx.common.creative.review.ReviewCreativeDao;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.*;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.impls.redis.JedisFetcher;
import com.bxm.warcar.cache.impls.redis.JedisUpdater;
import com.bxm.warcar.message.Message;
import com.bxm.warcar.message.dingding.DingDingMessageSender;
import com.bxm.warcar.utils.KeyBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.RandomUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author fgf
 * @date 2023/9/6
 **/
@Slf4j
@Configuration
public class ReviewCreativeBuilder implements AdxBidResponseBuilder {
    @Resource
    private ReviewCreativeDao reviewCreativeDao;

    @Resource
    private DingDingMessageSender dingDingMessageSender;

    @Resource
    private JedisFetcher fetcher;

    @Resource
    private JedisUpdater updater;

    @Override
    public void rebuildAdxBidResponse(BidResponse bidResponse, ResponseBuildAttribute attribute) {
        Dispatcher dispatcher = attribute.getDispatcher();
        List<ReviewCreative> creatives = getCreatives(dispatcher);
        if (CollectionUtils.isEmpty(creatives)) {
            return;
        }

        //处理素材
        for (SeatBid seatBid : bidResponse.getSeat_bid()) {
            for (Bid bid : seatBid.getBid()) {
                com.bxm.adx.common.sell.response.Native an = bid.getA_native();
                if (CollectionUtils.isNotEmpty(creatives) && Objects.nonNull(an) && CollectionUtils.isNotEmpty(an.getAssets())) {
                    List<ReviewCreative> reviewCreatives = queryReviewCreativeByProduct(creatives, bid,dispatcher);
                    ReviewCreative creative = randomCreative(reviewCreatives);
                    if (Objects.isNull(creative)) {
                        return;
                    }
                    replaceByReviewCreative(bid, creative);
                    bid.setCreate_id(creative.getId());
                    bid.setAdxCreateId(creative.getId());
                }
            }
        }
    }

    /**
     * 获取替换素材
     *
     * @param dispatcher
     * @return
     */
    private List<ReviewCreative> getCreatives(Dispatcher dispatcher) {
        List<ReviewCreative> creatives = queryReviewCreativeByDispatcher(dispatcher);
        return creatives;
    }

    /**
     * 随机出替换素材
     *
     * @param creatives
     * @return
     */
    private ReviewCreative randomCreative(List<ReviewCreative> creatives) {
        if (CollectionUtils.isNotEmpty(creatives)) {
            int r = RandomUtils.nextInt(creatives.size());
            return creatives.get(r);
        }
        return null;
    }

    /**
     * 根据launchProduct过滤出创意素材
     */
    private List<ReviewCreative> queryReviewCreativeByProduct(List<ReviewCreative> creatives,Bid bid,Dispatcher dispatcher) {
        String launchProduct = bid.getLaunchProduct();
        if (StringUtils.isBlank(launchProduct)) {
            return creatives;
        }
        List<ReviewCreative> creativesByProduct = creatives.stream().filter(c -> launchProduct.equals(c.getLaunchProduct())).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(creativesByProduct)) {
            //报警
            try {
                String fetch = fetcher.fetch(getKey(dispatcher,bid.getAdid()), String.class);
                if (StringUtils.isBlank(fetch)) {
                    dingDingMessageSender.send2(new Message(String.format("创意通知：投放产品【%s】未匹配到素材\n 广告位ID:%s \n 广告券ID:%s \n 广告平台ID:%s \n dsp广告位ID:%s \n dsp应用ID:%s", launchProduct, dispatcher.getPositionId(), bid.getAdid(), dispatcher.getDspId(), dispatcher.getDspPosid(), dispatcher.getDspAppid()), Arrays.asList("18322693791","15858242186")));
                    updater.update(getKey(dispatcher,bid.getAdid()),"1",60 * 3);
                }
            } catch (Exception e) {
                log.error(String.format("报警失败，投放产品【%s】未匹配到素材，positionId:%s - ticketId:%s - dspId:%s - dspPosid:%s - dspAppId:%s", launchProduct, dispatcher.getPositionId(), bid.getAdid(), dispatcher.getDspId(), dispatcher.getDspPosid(), dispatcher.getDspAppid()),e);
            }
            throw new AdxException(AdxErrEnum.CREATIVE_NOT_FOUND);
        }
        return creativesByProduct;
    }

    private KeyGenerator getKey(Dispatcher dispatcher,String adid) {
        return () -> KeyBuilder.build("DD_ALERT",dispatcher.getPositionId(), adid, dispatcher.getStrategyId());
    }

    /**
     * 获取所有替换素材
     *
     * @param dispatcher
     * @return
     */
    private List<ReviewCreative> queryReviewCreativeByDispatcher(Dispatcher dispatcher) {
        List<Long> creativeIds = dispatcher.getCreativeReviewIdList();
        if (CollectionUtils.isNotEmpty(creativeIds)) {
            List<ReviewCreative> creativeList = reviewCreativeDao.queryCreativeListByIds(creativeIds);
            return creativeList;
        }
        return null;
    }

    private void replaceByReviewCreative(Bid bid, ReviewCreative creative) {
        Native.NativeBuilder builder = Native.builder();
        List<Asset> assets = new ArrayList<>();

        String title = creative.getTitle();
        if (StringUtils.isNotBlank(title)) {
            assets.add(Asset.builder()
                    .type(AdxConstants.AssetType.TITLE.getType())
                    .text(Text.builder().text(title).build())
                    .build());
        }
        String content = creative.getContent();
        if (StringUtils.isNotBlank(content)) {
            assets.add(Asset.builder()
                    .type(AdxConstants.AssetType.CONTENT.getType())
                    .text(Text.builder().text(content).build())
                    .build());
        }
        String imgUrl = creative.getImgUrl();
        if (StringUtils.isNotBlank(imgUrl)) {
            assets.add(Asset.builder()
                    .type(AdxConstants.AssetType.LARGE_IMG.getType())
                    .img(Image.builder().url(imgUrl).w(creative.getImgWeight()).h(creative.getImgHeight()).build())
                    .build());
        }

        String iconUrl = creative.getIconUrl();
        if (StringUtils.isNotBlank(iconUrl)) {
            assets.add(Asset.builder()
                    .type(AdxConstants.AssetType.ICON.getType())
                    .img(Image.builder().url(iconUrl).w(creative.getIconWeight()).h(creative.getIconHeight()).build())
                    .build());
        }

        String videoUrl = creative.getVideoUrl();
        if (StringUtils.isNotBlank(videoUrl)) {
            Native bidNative = bid.getA_native();
            Asset videoAsset = bidNative.getAssets().stream().filter(asset -> Objects.nonNull(asset.getVideo())).findFirst().orElse(null);
            if (Objects.nonNull(videoAsset)) {
                Video bidVideo = videoAsset.getVideo();
                Video.VideoBuilder video = Video.builder();
                video.url(videoUrl)
                        .cover_url(creative.getCoverUrl())
                        .h(creative.getVideoHeight())
                        .w(creative.getVideoWeight())
                        .v_monitor(bidVideo.getV_monitor())
                        .duration(creative.getVideoDuration())
                ;
                videoAsset.setVideo(video.build());
                assets.add(videoAsset);
            }

        }

        builder.assets(assets);
        bid.setA_native(builder.build());
    }

    @Override
    public int getOrder() {
        return 0;
    }
}
