package com.bxm.adx.common.sell.position;

import com.bxm.adx.facade.constant.pushable.CachePushableFields;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.mccms.facade.model.pushable.PositionInteractAppentranceRefCacheVO;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.cache.push.Pushable;
import com.bxm.warcar.datasync.server.DataSyncProducer;
import com.bxm.warcar.integration.pushable.annotation.CachePush;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.Map;
import java.util.Objects;

/**
 * For cache pushable
 *
 * <pre>
 * name = "POSITION"
 * parameters = {positionId=$positionId}
 * byte[] = Position object to json bytes
 * </pre>
 *
 * @author allen
 * @since 2019-12-18
 */
@CachePush("POSITION")
@Component
@Slf4j
public class PositionDaoImpl implements PositionDao, Pushable {
    private final Fetcher fetcher;
    private final Updater updater;
    private final DataSyncProducer dataSyncProducer;
    private static final String APPID = "APPID";
    private static final String PID = "PID";
    private static final String TYPE = "TYPE";

    public PositionDaoImpl(Fetcher fetcher, Updater updater,
                           DataSyncProducer dataSyncProducer) {
        this.fetcher = fetcher;
        this.updater = updater;
        this.dataSyncProducer = dataSyncProducer;
    }

    @Override
    public PositionInteractAppentranceRefCacheVO getInteractByPositionId(String positionId) {
        PositionInteractAppentranceRefCacheVO cacheVO = fetcher.hfetch(AdxKeyGenerator.Position.getPositionInteract(), positionId, PositionInteractAppentranceRefCacheVO.class);
        return cacheVO;
    }

    @Override
    public void push(Map<String, Object> parameters, byte[] data) {
        if (log.isInfoEnabled()) {
            log.info("position data = {}", JsonHelper.convert(data, String.class));
        }
        Object o = parameters.get(CachePushableFields.POSITION_ID);
        if (Objects.isNull(o)) {
            return;
        }
        String positionId = Objects.toString(o);
        Position position = JsonHelper.convert(data, Position.class);

        String key = position.getClass().getName() + "$" + positionId;
        dataSyncProducer.post(key, position);
        //以下处理的redis数据不在ADX中使用，由于历史原因在ADX-SERVICE等其他项目中使用，因此不能直接修改
        updater.remove(getKeyGenerator(APPID, position.getAppPositionId()));
        updater.remove(getKeyGenerator(PID, positionId));

        String appId = position.getAppId();
        if (!StringUtils.isEmpty(appId)) {
            updater.srem(AdxKeyGenerator.App.getAllPositionIdFromApp(appId), positionId);
        }
        if (position.isEnabled()) {
            updater.update(getKeyGenerator(APPID, position.getAppPositionId()), position);
            updater.update(getKeyGenerator(PID, positionId), position);

            if (!StringUtils.isEmpty(appId)) {
                updater.supdate(AdxKeyGenerator.App.getAllPositionIdFromApp(appId), positionId);
            }
        }
    }

    // TODO com.bxm.adx.facade.constant.redis.AdxKeyGenerator.Position 这里也有定义
    private KeyGenerator getKeyGenerator(String name, String value) {
        return () -> KeyBuilder.build("ADX", "POSITION", name, value);
    }
}