package com.bxm.adx.common.sell.position.mapping;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.PositionMappingProperties;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.position.PositionChangeHandler;
import com.bxm.adx.common.sell.position.PositionService;
import com.bxm.adx.common.sell.request.Impression;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author fgf
 * @date 2023/2/22
 **/
@Component
public class PositionSizeStrategy implements PositionMappingStrategy, PositionChangeHandler {
    private final AdxProperties adxProperties;
    /**
     * 广告位尺寸类型-广告位集合
     * <pre>
     * |----------------------------|-----------------------------|
     * |    Key                     | Value                       |
     * |----------------------------|-----------------------------|
     * | {appId-positionScene-w*h}  |[position,position,...]      |
     * |----------------------------|-----------------------------|
     * </pre>
     */
    private final ConcurrentHashMap<String, Position> positionInfoMap = new ConcurrentHashMap<>();

    private final PositionService positionService;

    public PositionSizeStrategy(AdxProperties adxProperties, PositionService positionService) {
        this.adxProperties = adxProperties;
        this.positionService = positionService;
    }

    @Override
    public Strategy strategy() {
        return Strategy.Position_Size;
    }

    @Override
    public Position mapping(BidRequest bidRequest, Impression impression, PositionMappingProperties properties) {
        String appId = getAppId(bidRequest, properties);
        Position position = getPositionByInfo(appId, impression.getW(), impression.getH(), impression.getImp_type());
        return position;
    }

    /**
     * 获取复杂广告位映射（比如bes）
     *
     * @param appId
     * @param w
     * @param h
     * @param positionScene
     * @return
     */
    public Position getPositionByInfo(String appId, Integer w, Integer h, Integer positionScene) {
        if (StringUtils.isEmpty(appId)) {
            return null;
        }
        if (Objects.isNull(w) || Objects.isNull(h)) {
            return null;
        }
        if (Objects.isNull(positionScene)) {
            return null;
        }
        String key = buildMapKey(appId, w, h, positionScene);
        return positionInfoMap.get(key);
    }

    /**
     * 根据请求的os区别获取bes对应的appId
     *
     * @param bidRequest
     * @return
     */
    private String getAppId(BidRequest bidRequest, PositionMappingProperties properties) {
        if (Objects.nonNull(bidRequest.getDevice())) {
            if (bidRequest.getDevice().isIos()) {
                return properties.getIosAppId();
            }
        }
        return properties.getAndroidAppId();
    }

    @Override
    public void doUpdate(Position old, Position latest) {
        if (isExecuteUpdateAndDoDelete(old, latest)) {

            Strategy strategy = getStrategy(latest);
            switch (strategy) {
                case Position_Size:
                    String infoKey = buildMapKeyByPosition(latest);
                    if (StringUtils.isNotEmpty(infoKey)) {
                        positionInfoMap.put(infoKey, latest);
                    }
                    break;
            }
        }
    }

    @Override
    public void doDelete(Position old) {
        Strategy strategy = getStrategy(old);
        switch (strategy) {
            case Position_Size:
                String infoKey = buildMapKeyByPosition(old);
                if (StringUtils.isNotEmpty(infoKey)) {
                    positionInfoMap.remove(infoKey);
                }
                break;
        }
    }

    private Strategy getStrategy(Position position) {
        String appId = position.getAppId();
        List<PositionMappingProperties> positionMapping = adxProperties.getPositionMapping();
        if (CollectionUtils.isEmpty(positionMapping)) {
            return Strategy.Default;
        }
        for (PositionMappingProperties pm : positionMapping) {
            if (appId.equals(pm.getAndroidAppId())
                    || appId.equals(pm.getIosAppId())
                    || appId.equals(pm.getDefaultAppId())) {
                return pm.getStrategy();
            }
        }
        return Strategy.Default;
    }

    /**
     * 构建复杂广告位映射keu（比如bes）
     *
     * @param position
     * @return
     */
    public String buildMapKeyByPosition(Position position) {
        String appId = position.getAppId();
        String positionSize = position.getPositionSize();
        Byte scene = position.getPositionScene();
        if (StringUtils.isEmpty(appId)) {
            return null;
        }
        if (StringUtils.isEmpty(positionSize)) {
            return null;
        }
        if (Objects.isNull(scene)) {
            return null;
        }
        String size = positionSize.split(" ")[0];
        String val = new StringBuilder(appId)
                .append("-")
                .append(position.getPositionScene())
                .append("-")
                .append(size)
                .toString();
        return val;
    }

    /**
     * 构建复杂广告位映射keu（比如bes）
     *
     * @param appId
     * @param w
     * @param h
     * @param positionScene
     * @return
     */
    public String buildMapKey(String appId, int w, int h, int positionScene) {
        String val = new StringBuilder(appId)
                .append("-")
                .append(positionScene)
                .append("-")
                .append(w)
                .append("*")
                .append(h)
                .toString();
        return val;
    }
}
