package com.bxm.fossicker.base.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.fossicker.base.domain.CommonEquipmentAppMapper;
import com.bxm.fossicker.base.domain.EquipmentExtMapper;
import com.bxm.fossicker.base.domain.EquipmentMapper;
import com.bxm.fossicker.base.dto.EquipmentInfoDto;
import com.bxm.fossicker.base.enums.EquipmentEnum;
import com.bxm.fossicker.base.facade.model.EquipmentDTO;
import com.bxm.fossicker.base.param.AppInfoParam;
import com.bxm.fossicker.base.param.EquipmentBaseInfo;
import com.bxm.fossicker.base.param.EquipmentParam;
import com.bxm.fossicker.base.service.EquipmentService;
import com.bxm.fossicker.base.utils.EquipmentValidator;
import com.bxm.fossicker.base.vo.CommonEquipmentApp;
import com.bxm.fossicker.base.vo.EquipmentExtInfo;
import com.bxm.fossicker.base.vo.EquipmentInfo;
import com.bxm.fossicker.constant.UserRedisKeyConstant;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.vo.BasicParam;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.MD5Util;
import com.bxm.newidea.component.tools.StringUtils;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.retry.RetryException;
import org.springframework.retry.annotation.Backoff;
import org.springframework.retry.annotation.Retryable;
import org.springframework.stereotype.Service;

import java.lang.reflect.InvocationTargetException;
import java.util.*;
import java.util.stream.Collectors;

import static com.bxm.fossicker.base.constant.BaseRedisKey.EQUIPMENT_HASH;

@Service
public class EquipmentServiceImpl extends BaseService implements EquipmentService {

    private final EquipmentExtMapper equipmentExtMapper;

    private final EquipmentMapper equipmentMapper;

    private final UserInfoFacadeService userInfoFacadeService;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final DistributedLock distributedLock;

    private final CommonEquipmentAppMapper commonEquipmentAppMapper;

    private final static String DEFAULT_BLOCK_DEVID = "0";

    @Autowired
    public EquipmentServiceImpl(EquipmentExtMapper equipmentExtMapper,
                                EquipmentMapper equipmentMapper,
                                UserInfoFacadeService userInfoFacadeService,
                                RedisHashMapAdapter redisHashMapAdapter,
                                DistributedLock distributedLock,
                                CommonEquipmentAppMapper commonEquipmentAppMapper) {
        this.equipmentExtMapper = equipmentExtMapper;
        this.equipmentMapper = equipmentMapper;
        this.userInfoFacadeService = userInfoFacadeService;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.distributedLock = distributedLock;
        this.commonEquipmentAppMapper = commonEquipmentAppMapper;
    }


    @Override
    @Retryable(backoff = @Backoff(200), value = RetryException.class)
    public EquipmentInfoDto createMutedUser(EquipmentParam equipmentParam) {
        String requestId = nextSequence().toString();
        String key = equipmentParam.hashCode() + "";

        if (!distributedLock.lock(key, requestId)) {
            logger.error("创建设备信息存在重复请求:[{}]", equipmentParam);
            throw new RetryException("创建设备信息存在重复请求");
        }

        String id = equipmentParam.getDevcId();
        boolean existsEquipment = false;

        if (StringUtils.isNotBlank(id)
                && NumberUtils.isDigits(id)
                && !DEFAULT_BLOCK_DEVID.equals(id)) {
            //如果客户端传了设备唯一标识，就进行更新
            EquipmentDTO equipmentDTO = get(id);
            if (null != equipmentDTO) {
                this.createOrUpdate(equipmentParam, true, equipmentDTO);
                existsEquipment = true;
            } else {
                logger.error("请求提供了设备信息ID，但是数据库中并不存在对应的历史记录");
            }
        }

        if (!existsEquipment) {
            //如果客户端没有传唯一标识，或者传了唯一标识但是没有查到相应的设备信息，则录入设备信息
            //根据传来的设备id列表进行查询是否有已存在的设备唯一标识
            List<String> extractInfos = extractEquipment(equipmentParam);

            if (CollectionUtils.isEmpty(extractInfos)) {
                logger.error("参数中不包含任何设备信息,请求参数：{}", equipmentParam);
                return new EquipmentInfoDto(0L, DEFAULT_BLOCK_DEVID);
            }

            Long historyEquimentId = equipmentExtMapper.selectByIdentifiers(extractInfos);
            if (null != historyEquimentId) {
                //如果根据传来的设备信息能反向查到设备唯一标识则进行更新
                id = String.valueOf(historyEquimentId);
                equipmentParam.setDevcId(id);
                this.createOrUpdate(equipmentParam, true, null);
            } else {
                //如果根据传来的设备信息不能反向查到设备唯一标识则赋予新的设备唯一标识并进行录入
                id = nextSequence().toString();
                equipmentParam.setDevcId(id);
                this.createOrUpdate(equipmentParam, false, null);
            }
        }

        Long userId;
        // 如果userId为空，说明当前设备是新设备，则根据新设备进行原有的创建静默用户的逻辑
        if (Objects.isNull(equipmentParam.getUserId())) {
            userId = userInfoFacadeService.createMutedUser(Long.valueOf(id), equipmentParam);
            // 移除登录信息（因为可能会出现老用户卸载app之后，没有退出登录，根绝设备号创建用户返回的还是老用户的，
            // 后端有一部分逻辑是根据userId是否有token来判断是否登录）
            // 所以这里需要移除登录信息
            if (isBetterVersion(equipmentParam)) {
                logger.info("用户重新登录，注销token，相关参数：{}", equipmentParam);

                redisHashMapAdapter.remove(UserRedisKeyConstant.HASH_USER_TOKEN, Objects.toString(userId));
            }
        } else {
            // 否则直接使用客户端传过来的userId进行返回
            userId = equipmentParam.getUserId();
        }

        EquipmentInfoDto result = new EquipmentInfoDto();
        result.setEquipmentId(id);
        result.setUserId(userId);

        distributedLock.unlock(key, requestId);
        logger.debug("设备上报接口,设备与静默用户信息:[{}]", result);
        return result;
    }

    /**
     * 是否为更高的版本
     *
     * @param param 基础参数
     * @return true表示高版本
     */
    private boolean isBetterVersion(BasicParam param) {
        return null != param
                && null != param.getCurVer()
                && StringUtils.compareVersion(param.getCurVer(), "1.4.0") >= 0;
    }

    /**
     * 将设备号转换成一个列表方便查询
     *
     * @param equipmentParam 设备参数
     * @return 设备参数列表
     */
    private List<String> extractEquipment(EquipmentParam equipmentParam) {
        return new ArrayList<>(getEquipmentMap(equipmentParam).values());
    }

    /**
     * 将设备号转换成一个map方便查询
     *
     * @param equipmentParam 设备参数
     * @return 设备类型与对应的值映射
     */
    private Map<String, String> getEquipmentMap(EquipmentBaseInfo equipmentParam) {
        Map<String, String> map = new HashMap<>(5);

        if (StringUtils.isNoneEmpty(equipmentParam.getImei())) {
            map.put(EquipmentEnum.ANDROID_IMEI.name(), equipmentParam.getImei());
            map.put(EquipmentEnum.ANDROID_IMEI_MD5.name(), MD5Util.hgmd5(equipmentParam.getImei()));
        }

        if (StringUtils.isNotBlank(equipmentParam.getImeiMd5())) {
            map.put(EquipmentEnum.ANDROID_IMEI_MD5.name(), equipmentParam.getImeiMd5());
        }

        if (StringUtils.isNotBlank(equipmentParam.getAndroidId())) {
            map.put(EquipmentEnum.ANDROID_ID.name(), equipmentParam.getAndroidId());
            map.put(EquipmentEnum.ANDROID_ID_MD5.name(), MD5Util.hgmd5(equipmentParam.getAndroidId()));
        }

        if (StringUtils.isNotBlank(equipmentParam.getAndroidIdMd5())) {
            map.put(EquipmentEnum.ANDROID_ID_MD5.name(), equipmentParam.getAndroidIdMd5());
        }

        if (StringUtils.isNotBlank(equipmentParam.getAndroidUuid())) {
            map.put(EquipmentEnum.ANDROID_UUID.name(), equipmentParam.getAndroidUuid());
        }

        if (StringUtils.isNotBlank(equipmentParam.getIdfa()) && EquipmentValidator.idfa(equipmentParam.getIdfa())) {
            map.put(EquipmentEnum.IOS_IDFA.name(), equipmentParam.getIdfa());
            map.put(EquipmentEnum.IOS_IDFA_MD5.name(), MD5Util.hgmd5(equipmentParam.getIdfa()));
        }

        if (StringUtils.isNotBlank(equipmentParam.getIdfaMd5())) {
            map.put(EquipmentEnum.IOS_IDFA_MD5.name(), equipmentParam.getIdfaMd5());
        }

        if (StringUtils.isNotBlank(equipmentParam.getIosId())) {
            map.put(EquipmentEnum.IOS_DEVICE_ID.name(), equipmentParam.getIosId());
        }
        return map;
    }

    /**
     * 创建或更新设备相关信息
     *
     * @param equipmentParam 设备参数
     * @param hasHistory     是否存在历史记录
     */
    private void createOrUpdate(EquipmentParam equipmentParam, boolean hasHistory, EquipmentDTO existsEquipment) {
        Date now = new Date();

        //增加设备唯一标识信息
        List<EquipmentExtInfo> equipmentExtInfoList = new ArrayList<>();
        getEquipmentMap(equipmentParam).forEach((k, v) -> {
            EquipmentExtInfo equipmentExtInfo = EquipmentExtInfo.builder()
                    .createTime(now)
                    .modifyTime(now)
                    .id(Long.valueOf(equipmentParam.getDevcId()))
                    .identifierName(k)
                    .identifierValue(v)
                    .build();
            equipmentExtInfoList.add(equipmentExtInfo);
        });

        //增加设备额外信息
        EquipmentInfo equipmentInfo = new EquipmentInfo();
        org.springframework.beans.BeanUtils.copyProperties(equipmentParam, equipmentInfo);
        equipmentInfo.setEnableNotification(equipmentParam.isEnableNotification());
        equipmentInfo.setId(Long.valueOf(equipmentParam.getDevcId()));
        equipmentInfo.setModifyTime(now);
        equipmentInfo.setCurrentVersion(equipmentParam.getCurVer());

        //增加设备应用信息
        List<CommonEquipmentApp> commonEquipmentAppList = new ArrayList<>();
        if (StringUtils.isNotEmpty(equipmentParam.getAppInfoParamList())) {
            List<AppInfoParam> appInfoParamList = JSONObject.parseArray(equipmentParam.getAppInfoParamList(), AppInfoParam.class);
            commonEquipmentAppList = appInfoParamList
                    .stream().map(e -> {
                        CommonEquipmentApp commonEquipmentApp = new CommonEquipmentApp();
                        commonEquipmentApp.setAppName(e.getAppName());
                        commonEquipmentApp.setPackageName(e.getPackageName());
                        commonEquipmentApp.setId(Long.valueOf(equipmentParam.getDevcId()));
                        commonEquipmentApp.setModifyTime(new Date());
                        commonEquipmentApp.setCreateTime(new Date());
                        return commonEquipmentApp;
                    }).collect(Collectors.toList());
        }

        //如果存在历史记录则进行更新
        if (hasHistory) {
            compareAndModify(existsEquipment, equipmentInfo, equipmentExtInfoList, commonEquipmentAppList);
        } else {
            equipmentInfo.setCreateTime(now);
            equipmentMapper.insert(equipmentInfo);

            MybatisBatchBuilder.create(EquipmentExtMapper.class, equipmentExtInfoList).run(EquipmentExtMapper::insert);
            MybatisBatchBuilder.create(CommonEquipmentAppMapper.class, commonEquipmentAppList).run(CommonEquipmentAppMapper::insert);

            cleanCache(equipmentInfo.getId());
        }
    }

    private void cleanCache(Long equipmentId) {
        redisHashMapAdapter.remove(buildKey(equipmentId), String.valueOf(equipmentId));
    }

    /**
     * 对比历史设备信息和传入信息，如果无变化则不进行数据库操作
     * 只更新变动的部分，尽量减少数据库操作
     *
     * @param existsEquipment        已存在的设备信息
     * @param equipmentInfo          设备信息
     * @param equipmentExtInfoList   设备唯一信息
     * @param commonEquipmentAppList 设备安装列表信息
     */
    private void compareAndModify(EquipmentDTO existsEquipment,
                                  EquipmentInfo equipmentInfo,
                                  List<EquipmentExtInfo> equipmentExtInfoList,
                                  List<CommonEquipmentApp> commonEquipmentAppList) {
        if (null == existsEquipment) {
            existsEquipment = get(String.valueOf(equipmentInfo.getId()));
        }

        boolean hit = false;

        // 比对设备基础信息(操作系统、安装系统版本、是否开启通知)
        EquipmentInfo existsInfo = parseInfo(existsEquipment);
        if (!equipmentInfo.equals(existsInfo)) {
            equipmentMapper.updateByPrimaryKey(equipmentInfo);

            hit = true;
            logger.debug("设备基础信息变更：{}", equipmentInfo);
        }

        // 比对设备唯一标识信息
        List<EquipmentExtInfo> existsExtInfos = parseExtInfo(existsEquipment);
        if (isNotEquals(existsExtInfos, equipmentExtInfoList)) {
            equipmentExtMapper.delete(equipmentInfo.getId());
            MybatisBatchBuilder.create(EquipmentExtMapper.class, equipmentExtInfoList).run(EquipmentExtMapper::insert);

            hit = true;
            logger.debug("设备标识信息变更：{},唯一标识：{}", equipmentInfo, equipmentExtInfoList);
        }

        //对比设备安装列表
        if (isNotEquals(existsEquipment.getApps(), commonEquipmentAppList)) {
            commonEquipmentAppMapper.delete(equipmentInfo.getId());
            MybatisBatchBuilder.create(CommonEquipmentAppMapper.class, commonEquipmentAppList).run(CommonEquipmentAppMapper::insert);

            hit = true;
            logger.debug("设备安装信息列表变更：{}", equipmentInfo);
        }

        if (hit) {
            cleanCache(equipmentInfo.getId());
        } else {
            logger.debug("设备[{}]信息未变更，不做更新处理", equipmentInfo.getId());
        }
    }

    private boolean isNotEquals(List source, List target) {
        if (null == source && null == target) {
            return false;
        }

        if (null == source || null == target) {
            return true;
        }

        return !CollectionUtils.isEqualCollection(source, target);
    }

    private EquipmentInfo parseInfo(EquipmentDTO existsEquipment) {
        return EquipmentInfo.builder()
                .id(Long.valueOf(existsEquipment.getId()))
                .currentVersion(existsEquipment.getCurVer())
                .operatingSystem(existsEquipment.getOperatingSystem())
                .enableNotification(existsEquipment.isEnableNotification())
                .phoneModel(existsEquipment.getPhoneModel())
                .build();
    }

    private List<EquipmentExtInfo> parseExtInfo(EquipmentDTO existsEquipment) {
        List<EquipmentExtInfo> equipmentExtInfoList = new ArrayList<>();
        getEquipmentMap(existsEquipment).forEach((k, v) -> {
            EquipmentExtInfo equipmentExtInfo = EquipmentExtInfo.builder()
                    .identifierName(k)
                    .identifierValue(v)
                    .build();
            equipmentExtInfoList.add(equipmentExtInfo);
        });
        return equipmentExtInfoList;
    }

    private KeyGenerator buildKey(Long id) {
        return EQUIPMENT_HASH.copy().appendKey(id % 20);
    }

    @Override
    public EquipmentDTO get(String equipmentId) {
        //特殊设备号，或者非数字的设备号，均认为非法
        if (DEFAULT_BLOCK_DEVID.equals(equipmentId) || !NumberUtils.isDigits(equipmentId)) {
            logger.error("非法的设备号：{}", equipmentId);
            return null;
        }

        Long id = Long.valueOf(equipmentId);
        KeyGenerator key = buildKey(id);

        EquipmentDTO equipmentDTO = redisHashMapAdapter.get(key, equipmentId, EquipmentDTO.class);

        if (null == equipmentDTO) {
            equipmentDTO = new EquipmentDTO();
            List<EquipmentExtInfo> equipmentExtInfoList = equipmentExtMapper.selectByPrimaryKey(id);
            if (CollectionUtils.isNotEmpty(equipmentExtInfoList)) {
                //转换为map
                Map<String, String> map = new HashMap<>(3);
                equipmentExtInfoList.forEach(e ->
                        map.put(EquipmentEnum.valueOf(e.getIdentifierName()).getType(), e.getIdentifierValue()));
                try {
                    //转换为对象
                    BeanUtils.populate(equipmentDTO, map);
                    equipmentDTO.setId(equipmentId);
                } catch (IllegalAccessException | InvocationTargetException e) {
                    logger.error(e.getMessage(), e);
                }
            }

            //设备安装列表
            equipmentDTO.setApps(commonEquipmentAppMapper.getEquipmentApps(id));

            redisHashMapAdapter.put(key, equipmentId, equipmentDTO);
        }

        if (equipmentDTO.getId() == null) {
            equipmentDTO = null;
        }

        return equipmentDTO;
    }

    /**
     * 用户是否开启推送
     */
    @Override
    public boolean checkHasOpenPush(String deviceId) {
        if (!NumberUtils.isDigits(deviceId)) {
            logger.error("请求设备ID不是一个数字，非法请求：{}", deviceId);
            return false;
        }
        EquipmentInfo equipmentInfo = equipmentMapper.selectByPrimaryKey(Long.valueOf(deviceId));
        return null != equipmentInfo && equipmentInfo.getEnableNotification();
    }
}
