package com.bxm.fossicker.base.service.impl;

import com.bxm.fossicker.base.config.BaseInfoPeroperties;
import com.bxm.fossicker.base.param.CommodityShareParam;
import com.bxm.fossicker.base.service.ShortLinkService;
import com.bxm.fossicker.common.constant.BaseRedisKey;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
@Slf4j
public class ShortLinkServiceImpl implements ShortLinkService {

    /**
     * 基础信息配置服务
     */
    private final BaseInfoPeroperties baseInfoPeroperties;

    /**
     * 短链接host
     */
    private final static String REDIRECT = "r";

    private final RedisStringAdapter redisStringAdapter;

    @Autowired
    public ShortLinkServiceImpl(BaseInfoPeroperties baseInfoPeroperties, RedisStringAdapter redisStringAdapter) {
        this.baseInfoPeroperties = baseInfoPeroperties;
        this.redisStringAdapter = redisStringAdapter;
    }

    @Override
    public String getCommodityShareShortLink(CommodityShareParam commodityShareParam) {
        log.debug("生成分享链接，请求参数[{}]", commodityShareParam);

        String shareBaseUrl = baseInfoPeroperties.getShareCommodityUrl();

        shareBaseUrl = assembleShareCommodityUrl(shareBaseUrl, commodityShareParam);
        return getShortUrl(shareBaseUrl, true);
    }

    @Override
    public String getCommodityTicketShortLink(CommodityShareParam commodityShareParam) {

        String shareBaseUrl = baseInfoPeroperties.getTicketCommodityUrl();

        shareBaseUrl = assembleShareCommodityUrl(shareBaseUrl, commodityShareParam);
        return getShortUrl(shareBaseUrl, true);
    }

    @Override
    public String generateShortUrl(String url) {
        return getShortUrl(url, true);
    }

    /**
     * 组装商品分享url
     *
     * @param baseUrl             基础url
     * @param commodityShareParam 商品分享参数
     * @return 实际请求url
     */
    private String assembleShareCommodityUrl(String baseUrl, CommodityShareParam commodityShareParam) {

        StringBuilder stringBuilder = new StringBuilder(baseUrl);
        if (null != commodityShareParam.getGoodsId()) {
            stringBuilder.append("goodsId=").append(commodityShareParam.getGoodsId().toString());
        }

        if (null != commodityShareParam.getShareUserId()) {
            stringBuilder.append("&userId=").append(commodityShareParam.getShareUserId().toString());
        }

        if (null != commodityShareParam.getRelationId()) {
            stringBuilder.append("&relationId=").append(commodityShareParam.getRelationId());
        }

        return stringBuilder.toString();
    }

    /**
     * 获取短链,供网关做转发的短链
     *
     * @param url     实际请求的url
     * @param expired 是否需要缓存
     * @return 生成的短链
     */
    private String getShortUrl(String url, boolean expired) {
        log.debug("长链转短链，原始链接为[{}]，是否会过期[{}]", url, expired);

        String shortCode = StringUtils.longToString(url.hashCode());

        //短连接地址
        StringBuilder sortUrl = new StringBuilder();
        sortUrl.append(baseInfoPeroperties.getServerHost()).append("/").append(REDIRECT).append("/");
        sortUrl.append(getDiffCode(shortCode, url, expired));

        log.debug("长链转短链完成，原始链接为[{}]，转化后的短链为[{}]，是否会过期[{}]", url, sortUrl, expired);
        return sortUrl.toString();
    }

    /**
     * 防止碰撞，判断是否在缓存中已经存在对应的短码
     * 递归进行判断，直到和现有的缓存键值不冲突为止
     *
     * @param shortCode 原始短码
     * @param sourceUrl 原始链接
     * @param expired   是否过期
     * @return 新的短码
     */
    private String getDiffCode(String shortCode, String sourceUrl, boolean expired) {
        KeyGenerator shortLinkKey = BaseRedisKey.COMMODITY_SHARE_URL.copy().appendKey(shortCode);

        String cacheUrl = redisStringAdapter.get(shortLinkKey, String.class);
        if (null != cacheUrl) {
            if (StringUtils.equals(cacheUrl, sourceUrl)) {
                return shortCode;
            }

            log.info("短链已经存在，短链编码:[{}],缓存链接:[{}],原始链接:[{}]", shortCode, cacheUrl, sourceUrl);
            return getDiffCode(shortCode + "1", sourceUrl, expired);
        }

        if (expired) {
            redisStringAdapter.set(shortLinkKey, sourceUrl, 3600 * 24 * 30);
        } else {
            redisStringAdapter.set(shortLinkKey, sourceUrl);
        }

        return shortCode;
    }
}
