package com.bxm.fossicker.base.service.impl.popup;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.base.bo.FilterPopUpWindowsBO;
import com.bxm.fossicker.base.bo.UserPopUpedBO;
import com.bxm.fossicker.base.constant.PopUpRedisConstant;
import com.bxm.fossicker.base.domain.CommonPopUpWindowsMapper;
import com.bxm.fossicker.base.entity.CommonPopUpWindowsEntry;
import com.bxm.fossicker.base.enums.PopUpEnum;
import com.bxm.fossicker.base.param.PopUpWindowsClickParam;
import com.bxm.fossicker.base.param.PopUpWindowsCloseParam;
import com.bxm.fossicker.base.param.PopUpWindowsListParam;
import com.bxm.fossicker.base.service.PopUpWindowsService;
import com.bxm.fossicker.base.service.impl.popup.annotation.PopUpProcess;
import com.bxm.fossicker.base.service.impl.popup.interfacies.ClosePopUp;
import com.bxm.fossicker.base.vo.PopUpWindowsVO;
import com.bxm.fossicker.constant.UserRedisKeyConstant;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.aop.support.AopUtils;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.RequestBody;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 弹窗相关接口实现
 * @Author: Gonzo
 * @date 2019-09-02 18:32
 */
@Service
@Slf4j
public class PopUpWindowsServiceImpl implements PopUpWindowsService {

    @Resource
    private PopUpWindowsInterceptorChain interceptorChain;

    @Resource
    private CommonPopUpWindowsMapper commonPopUpWindowsMapper;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    private static final Map<PopUpEnum, ClosePopUp> CLOSE_POP_UP_MAP = Maps.newHashMap();

    @Override
    public List<PopUpWindowsVO> list(PopUpWindowsListParam param) {

        // 从缓存拿
        List<CommonPopUpWindowsEntry> commonPopUpWindowsEntries = loadPopUpWindowsCache(param.getPosition());

        // 判断用户是否登录
        String token = redisHashMapAdapter.get(UserRedisKeyConstant.HASH_USER_TOKEN, Objects.toString(param.getUserId()),
                String.class);

        // 获取用户 or 设备弹窗数据的缓存
        KeyGenerator key = PopUpRedisConstant.USER_POPUP_CACHE.copy()
                .appendKey(Objects.toString(StringUtils.isBlank(token)
                        ? StringUtils.join("DEVICE_", param.getDevcId()) : param.getUserId()));

        List<UserPopUpedBO> userPopUpEds = redisHashMapAdapter.values(key, UserPopUpedBO.class);
        Map<String, UserPopUpedBO> popUpEdInfo = Maps.newHashMap();

        if (!CollectionUtils.isEmpty(userPopUpEds)) {
            popUpEdInfo = userPopUpEds.stream()
                    .collect(Collectors.toMap(p -> Objects.toString(p.getPopUpId()), p -> p));
        }


        // 执行过滤，数据填充处理器
        interceptorChain.invoke(FilterPopUpWindowsBO.builder()
                .param(param)
                .popUpWindows(commonPopUpWindowsEntries)
                .popUpedInfo(popUpEdInfo)
                .build());


        // 处理完成后，更新用户弹出数据缓存
        redisHashMapAdapter.putAll(key, popUpEdInfo);

        // 转换数据 并返回
        return commonPopUpWindowsEntries.stream()
                .map(this::convert)
                .collect(Collectors.toList());
    }

    /**
     * 获取弹窗数据，优先从缓存中获取
     * @return 弹窗数据
     */
    private List<CommonPopUpWindowsEntry> loadPopUpWindowsCache(Byte position) {


        List<CommonPopUpWindowsEntry> commonPopUpWindowsEntries;
        String cacheStr = redisHashMapAdapter.get(PopUpRedisConstant.POP_UP_POSITION_CACHE, Objects.toString(position),
                String.class);

        if (StringUtils.isBlank(cacheStr)) {
            commonPopUpWindowsEntries = commonPopUpWindowsMapper.selectByPositionAndPlatform(position);

            redisHashMapAdapter.put(PopUpRedisConstant.POP_UP_POSITION_CACHE, Objects.toString(position),
                    JSON.toJSONString(commonPopUpWindowsEntries));

        } else {
            commonPopUpWindowsEntries = JSON.parseArray(cacheStr, CommonPopUpWindowsEntry.class);
        }

        return commonPopUpWindowsEntries;
    }

    /**
     * 转换对象
     * @param commonPopUpWindowsEntry 数据库对象
     * @return vo对象
     */
    private PopUpWindowsVO convert(CommonPopUpWindowsEntry commonPopUpWindowsEntry) {
        return PopUpWindowsVO.builder()
                .id(commonPopUpWindowsEntry.getId())
                .name(commonPopUpWindowsEntry.getName())
                .type(commonPopUpWindowsEntry.getType())
                .url(commonPopUpWindowsEntry.getJumpUrl())
                .imgUrl(commonPopUpWindowsEntry.getImgUrl())
                .width(commonPopUpWindowsEntry.getWidth())
                .height(commonPopUpWindowsEntry.getHeight())
                .ext(commonPopUpWindowsEntry.getExt())
                .build();
    }


    @Override
    public Message close(@RequestBody PopUpWindowsCloseParam param) {
        // 因为只有功能性以及系统性弹窗才需要点击关闭，所以这里关闭时，肯定有type。
        if (Objects.isNull(param.getType())) {
            log.warn("type 为空");
            return Message.build();
        }

        PopUpEnum byType = PopUpEnum.getByType(param.getType());
        if (Objects.isNull(byType)) {
            // 客户端好像总是上报一些不存在的type 避免warn日志的杂乱 这里区分下
            if (log.isDebugEnabled()) {
                log.debug("type: {} 属于未定义的弹窗关闭类型", param.getType());
            }
            return Message.build();
        }

        ClosePopUp closePopUp = CLOSE_POP_UP_MAP.get(byType);
        if (Objects.isNull(closePopUp)) {
            log.warn("type: {} 获取不到对应的处理关闭类", param.getType());
            return Message.build();
        }

        // 关闭
        closePopUp.close(param);
        return Message.build();
    }

    @Override
    public Message click(@RequestBody PopUpWindowsClickParam param) {
        return  Message.build();
    }


    @EventListener(ContextRefreshedEvent.class)
    public void onApplicationStartedEvent(ContextRefreshedEvent event) {
        Collection<ClosePopUp> closePopUps = event.getApplicationContext().getBeansOfType(ClosePopUp.class).values();

        closePopUps.forEach(p -> {
            PopUpProcess popUpProcess = AnnotationUtils.getAnnotation(AopUtils.getTargetClass(p),
                    PopUpProcess.class);

            // 设置处理关闭的类
            if (Objects.nonNull(popUpProcess)
                    && !Objects.equals(popUpProcess.special(), PopUpEnum.NONE)) {
                CLOSE_POP_UP_MAP.put(popUpProcess.special(), p);
            }
        });
    }
}
