package com.bxm.fossicker.base.service.impl.popup.interceptor;

import com.alibaba.fastjson.JSON;
import com.bxm.fossicker.base.bo.FilterPopUpWindowsBO;
import com.bxm.fossicker.base.entity.CommonPopUpWindowsEntry;
import com.bxm.fossicker.base.service.impl.popup.AbstractPopUpWindowsInterceptor;
import com.bxm.fossicker.base.service.impl.popup.annotation.PopUpProcess;
import com.bxm.fossicker.order.facade.service.OrderFacadeService;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.user.facade.enums.UserTypeEn;
import com.bxm.fossicker.user.facade.enums.VipTypeEnum;
import com.bxm.fossicker.user.facade.vip.VipFacadeService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 用户类型过滤器
 * @author Gonzo
 * @date 2019-10-10 14:07
 */
@PopUpProcess
public class UserTypePopUpWindowsInterceptor extends AbstractPopUpWindowsInterceptor {

    @Autowired
    private UserInfoFacadeService userInfoFacadeService;

    @Autowired
    private OrderFacadeService orderFacadeService;

    @Autowired
    private VipFacadeService vipFacadeService;

    @Override
    protected void doInvoke(FilterPopUpWindowsBO popUpWindows) {
        List<CommonPopUpWindowsEntry> commonPopUpWindowsEntryList = popUpWindows.getPopUpWindows();

        // 过滤用户类型
        List<CommonPopUpWindowsEntry> removeList = commonPopUpWindowsEntryList.stream().filter(p -> {

            // 返回是否需要过滤

            // 0：全部用户 1：新用户 2：老用户 3：未下单用户 4：已下单用户 5：自定义用户id 6：未开卡用户（未购买终身黑钻卡的用户）
            // 7：非会员用户（非终身会员和有有效期会员的用户） 8：（会员用户：包括月卡在有效期内的用户以及领取黑钻卡的用户）

            if (Objects.isNull(p.getUserType())) {
                return true;
            }

            // 所有用户 不过滤
            if (p.getUserType().intValue() == 0) {
                return false;
            }

            // userId 为null 直接过滤
            if (Objects.isNull(popUpWindows.getParam().getUserId())) {
                return true;
            }

            switch (p.getUserType()) {
                case UserTypeEn.NOVICE:
                    // 新用户过滤弹出 则非新用户过滤
                    return !userInfoFacadeService.isNoviceUser(popUpWindows.getParam().getUserId());
                case UserTypeEn.OLD:
                    // 老用户弹出 则新用户过滤
                    return userInfoFacadeService.isNoviceUser(popUpWindows.getParam().getUserId());
                case UserTypeEn.NOT_ORDER:
                    // 未下单用户弹出 则已下单用户过滤
                    return orderFacadeService.hasOrdered(popUpWindows.getParam().getUserId());
                case UserTypeEn.ORDERED:
                    // 已下单用户弹出 则未下单用户过滤
                    return !orderFacadeService.hasOrdered(popUpWindows.getParam().getUserId());
                case UserTypeEn.CUSTOM:
                    List<Long> userIds = JSON.parseArray(p.getUserIds(), Long.class);

                    if (!CollectionUtils.isEmpty(userIds)) {
                        // 指定用户id 如果不包含在内 则过滤
                        return !userIds.contains(popUpWindows.getParam().getUserId());
                    }
                    // 默认过滤
                    return true;
                case UserTypeEn.NOT_LIFELONG_VIP:
                    // 未开卡用户（未购买终身黑钻卡的用户）弹出 如果是开卡用户（购买终身黑钻卡的用户）则过滤
                    Byte vipStatus = vipFacadeService.getVipStatus(popUpWindows.getParam().getUserId());
                    return Objects.equals(vipStatus, (byte) VipTypeEnum.VIP_FOREVER.getType());
                case UserTypeEn.NOT_VIP:
                    // 非会员用户才可弹出 则是会员就过滤
                    return vipFacadeService.isVip(popUpWindows.getParam().getUserId());
                case UserTypeEn.VIP:
                    // 会员用户才可弹出 则非会员就过滤
                    return !vipFacadeService.isVip(popUpWindows.getParam().getUserId());
                default:
                    ;
            }

            return true;
        }).peek(p -> {
            if (log.isDebugEnabled()) {
                log.debug("用户类型过滤器，过滤弹窗: {}", p.getName());
            }
        }).collect(Collectors.toList());


        commonPopUpWindowsEntryList.removeAll(removeList);
    }


}
