package com.bxm.fossicker.base.service.impl.popup.process;

import com.bxm.fossicker.base.bo.FilterPopUpWindowsBO;
import com.bxm.fossicker.base.bo.UserPopUpedBO;
import com.bxm.fossicker.base.constant.PopUpRedisConstant;
import com.bxm.fossicker.base.domain.CommonPopUpWindowsMapper;
import com.bxm.fossicker.base.entity.CommonPopUpWindows;
import com.bxm.fossicker.base.entity.CommonPopUpWindowsEntry;
import com.bxm.fossicker.base.enums.PopUpEnum;
import com.bxm.fossicker.base.param.PopUpWindowsCloseParam;
import com.bxm.fossicker.base.param.PopUpWindowsListParam;
import com.bxm.fossicker.base.service.PopUpWindowsService;
import com.bxm.fossicker.base.service.impl.popup.AbstractPopUpWindowsInterceptor;
import com.bxm.fossicker.base.service.impl.popup.interfacies.ClosePopUp;
import com.bxm.fossicker.base.service.impl.popup.interfacies.PopUpWindowsProcess;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 集弹窗拦截器、弹窗数据填充 为一体的处理类
 *
 * @author Gonzo
 * @date 2019-10-10 14:10
 */
public abstract class AbstractPopUpWindowsProcess extends AbstractPopUpWindowsInterceptor implements ClosePopUp,
        PopUpWindowsProcess {

    @Autowired
    protected RedisHashMapAdapter redisHashMapAdapter;

    @Autowired
    protected CommonPopUpWindowsMapper commonPopUpWindowsMapper;

    protected PopUpEnum currentSupportPopUpKey;

    /**
     * 当前处理器支持的弹窗key
     */
    protected static final String CURRENT_PROCESS_SUPPORT_POP_UP_KEY = "CURRENT_PROCESS_SUPPORT_POP_UP_KEY";


    @Override
    protected void doInvoke(FilterPopUpWindowsBO popUpWindows) {

        if (preInvoke(popUpWindows)) {
            postInvoke(popUpWindows);
            afterCompletion(popUpWindows);
        }
    }

    @Override
    public void close(PopUpWindowsCloseParam param) {
        // 弹窗关闭事件记录
        doClose(param);
    }

    @Override
    public void setProcessSupportPopUp(PopUpEnum popUp) {
        // 设置当前类支持处理的弹窗类型
        this.currentSupportPopUpKey = popUp;
    }

    /**
     * 判断弹窗集合里是否有当前处理器支持的弹窗，并保存到对象中，用作数据处理时获取
     *
     * @param popUpWindows 包含弹窗数据的对象
     * @param popUp        判断是否存在的类型
     *
     * @return true 存在
     */
    private boolean didHaveCurrentProcessSupportPopUp(FilterPopUpWindowsBO popUpWindows, PopUpEnum popUp) {

        if (!CollectionUtils.isEmpty(popUpWindows.getPopUpWindows()) && Objects.nonNull(popUp)) {
            List<CommonPopUpWindowsEntry> currentSupport = getByType(popUpWindows.getPopUpWindows(), popUp);

            if (!CollectionUtils.isEmpty(currentSupport)) {
                popUpWindows.getParams().put(CURRENT_PROCESS_SUPPORT_POP_UP_KEY, currentSupport);
                return true;
            }
        }

        return false;
    }

    /**
     * 根据类型获取对应的弹窗数据
     * @param popUpWindows 包含弹窗数据的对象
     * @param popUp 类型
     * @return 弹窗数据
     */
    private List<CommonPopUpWindowsEntry> getByType(List<CommonPopUpWindowsEntry> popUpWindows, PopUpEnum popUp) {

        if (!CollectionUtils.isEmpty(popUpWindows)) {
            return popUpWindows.stream()
                    .filter(p -> Objects.equals(p.getType(), popUp.getType()))
                    .collect(Collectors.toList());
        }

        return Lists.newArrayList();
    }

    /**
     * 关闭弹窗，增加弹出次数
     * 这个方法只可以被点击上报事件所触发
     *  @see #close
     * 因为会即时更新缓存
     *
     * 如果需要在处理弹窗数据时
     *  @see #doInvoke
     * 更改弹窗的弹出信息，请调用
     *  @see #autoClose
     *
     * @param popUpId 弹窗id
     * @param unique userId or deviceId
     */
    protected void closeAndIncrementTimes(Long popUpId, String unique) {

        KeyGenerator key = PopUpRedisConstant.USER_POPUP_CACHE.copy().appendKey(unique);
        UserPopUpedBO userPopUpedBO = redisHashMapAdapter.get(key,
                Objects.toString(popUpId), UserPopUpedBO.class);

        CommonPopUpWindows commonPopUpWindows = commonPopUpWindowsMapper.selectByPrimaryKey(popUpId);

        // 如果为空则创建用户弹出次数记录对象
        if (Objects.isNull(userPopUpedBO) && Objects.nonNull(commonPopUpWindows)) {
            userPopUpedBO = new UserPopUpedBO();
            userPopUpedBO.setDay(new Date());
            userPopUpedBO.setTimes(0);
            userPopUpedBO.setPopUpId(popUpId);
            userPopUpedBO.setType(commonPopUpWindows.getType());
            userPopUpedBO.setCategory(commonPopUpWindows.getCategory());
            // 不是永久关闭
            userPopUpedBO.setFinalClose(Boolean.FALSE);
        } else {
            // 如果不为空，则说明当前弹窗有历史弹出数据，更新日期到今日即可
            userPopUpedBO.setDay(new Date());
        }

        userPopUpedBO.setTimes(Objects.isNull(userPopUpedBO.getTimes()) ? 1: userPopUpedBO.getTimes() + 1);

        // 更新缓存
        redisHashMapAdapter.put(key, Objects.toString(popUpId), userPopUpedBO);
    }

    /**
     * 获取弹窗id、如果已经传了id直接返回id，否则根据type获取id
     * @param param 弹窗关闭id
     * @return 弹窗id
     */
    protected Long getPopId(PopUpWindowsCloseParam param) {
        Long popId;
        if (Objects.nonNull(param.getPopUpId())) {
            popId = param.getPopUpId();

        } else {
            // 一般都不会到这里，新版本之后都是会传id参数
            popId = getPopIdByType(param.getType());
        }

        return popId;
    }

    /**
     * 根据弹窗类型获取弹窗的id
     * @param type 弹窗类型
     * @return 弹窗id
     */
    protected Long getPopIdByType(Byte type) {

        List<CommonPopUpWindows> commonPopUpWindows = commonPopUpWindowsMapper.selectByType(type);

        if (!CollectionUtils.isEmpty(commonPopUpWindows)) {
            return commonPopUpWindows.get(0).getId();
        }

        return null;
    }

    /**
     * 移除指定类型的弹窗，同时移除 CURRENT_PROCESS_SUPPORT_POP_UP_KEY 对应的弹窗 避免after代码处理
     *
     * @param popUp 弹窗类型
     */
    protected void remove(FilterPopUpWindowsBO popUpWindows, PopUpEnum popUp) {

        // 基础集合里的弹窗数据
        if (!CollectionUtils.isEmpty(popUpWindows.getPopUpWindows()) && !Objects.isNull(popUp)) {
            popUpWindows.getPopUpWindows().removeIf(p -> Objects.equals(p.getType(), popUp.getType()));
        }

        // 同时移除当前处理的弹窗数据
        popUpWindows.getParams().remove(CURRENT_PROCESS_SUPPORT_POP_UP_KEY);
    }

    /**
     * 移除当前处理类处理的弹窗
     * @param popUpWindows 请求参数
     */
    protected void removeCurrent(FilterPopUpWindowsBO popUpWindows) {
        remove(popUpWindows, this.currentSupportPopUpKey);
    }

    /**
     * 关闭弹窗触发
     * 本方法只适合弹窗数据处理时调用
     *      @see #doInvoke
     * 因为只会从参数中获取当前弹窗的弹出信息，并处理，而不会即时更新，而是在
     *      @see PopUpWindowsService#list(PopUpWindowsListParam)  方法中进行统一的缓存更新
     * @param popUpWindows popUpWindows
     * @param finalClose 是否永久关闭 true: 永久关闭 false: 普通关闭
     * @param times 要增加的弹出次数 有些关闭是不需要增加弹出次数的，例如某些任务发现完成之后就直接关闭了一次也没弹
     */
    protected void autoClose(FilterPopUpWindowsBO popUpWindows, Boolean finalClose, int times) {

        List<CommonPopUpWindowsEntry> commonPopUpWindowsEntryList = popUpWindows.getParam(CURRENT_PROCESS_SUPPORT_POP_UP_KEY);

        if (!CollectionUtils.isEmpty(commonPopUpWindowsEntryList)) {
            commonPopUpWindowsEntryList.forEach(p -> {

                String key = Objects.toString(p.getId());
                UserPopUpedBO userPopUpedBO = popUpWindows.getPopUpedInfo().get(key);

                // 如果为空 则创建用户的弹窗数据
                if (Objects.isNull(userPopUpedBO)) {
                    userPopUpedBO = new UserPopUpedBO();
                    userPopUpedBO.setDay(new Date());
                    userPopUpedBO.setTimes(0);
                    userPopUpedBO.setPopUpId(p.getId());
                    userPopUpedBO.setType(p.getType());
                    userPopUpedBO.setCategory(p.getCategory());
                    userPopUpedBO.setFinalClose(finalClose);

                    popUpWindows.getPopUpedInfo().put(key, userPopUpedBO);

                } else {
                    userPopUpedBO.setDay(new Date());
                }

                // 增加次数
                userPopUpedBO.setTimes(Objects.isNull(userPopUpedBO.getTimes())
                        ? times : userPopUpedBO.getTimes() + times);

                // 这里不能直接更新缓存，而是在外面直接统一更新 具体查看方法说明里的方法
            });
        }
    }


    /**
     * 执行前的处理
     * 默认会判断并获取当前处理类所支持的弹窗对象，如果不存在则校验不通过 从而终止之后的弹窗处理
     *
     * @param popUpWindows popUpWindows
     * @return false 跳过处理
     */
    protected boolean preInvoke(FilterPopUpWindowsBO popUpWindows) {

        if (Objects.isNull(currentSupportPopUpKey)) {
            log.warn("当前处理类: {} 没有设置好支持的弹窗类型", this.getClass());
            return false;
        }

        return didHaveCurrentProcessSupportPopUp(popUpWindows, currentSupportPopUpKey);
    }

    /**
     * 过滤不符合规则的弹窗
     *
     * @param popUpWindows 包含弹窗数据等信息对象
     */
    protected abstract void postInvoke(FilterPopUpWindowsBO popUpWindows);


    /**
     * 关闭弹窗事件
     *
     * @param param 关闭参数
     */
    protected abstract void doClose(PopUpWindowsCloseParam param);

    /**
     * 执行后的处理
     * 一般用来弹窗数据填充
     * @param popUpWindows 包含弹窗数据等信息对象
     */
    protected void afterCompletion(FilterPopUpWindowsBO popUpWindows) {

    }

}
